# -*- coding: utf-8 -*-

import itertools
import logging

import sandbox.projects.common.dynamic_models.compare as models_compare
import sandbox.projects.common.dynamic_models.utils as models_utils
from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.task import SandboxTask


class Archive1(sp.ResourceSelector):
    name = 'archive_1'
    description = 'archive 1'
    resource_type = ['DYNAMIC_MODELS_ARCHIVE', 'DYNAMIC_MODELS_ARCHIVE_BASE']
    required = True


class Archive2(sp.ResourceSelector):
    name = 'archive_2'
    description = 'archive 2'
    resource_type = ['DYNAMIC_MODELS_ARCHIVE', 'DYNAMIC_MODELS_ARCHIVE_BASE']
    required = True


class StrictMode(sp.SandboxBoolParameter):
    name = 'strict_mode'
    description = 'fail if production models changed'
    default_value = True


class CompareDynamicModels(SandboxTask):
    """
        Сравнить два архива с моделями и построить табличку
        с добавленными, удаленными, измененными и нетронутыми моделями
    """

    type = 'COMPARE_DYNAMIC_MODELS'

    input_parameters = (
        Archive1,
        Archive2,
        StrictMode,
    )

    @property
    def footer(self):
        return models_compare.generate_diff_footer(self.ctx.get('diff'))

    def on_execute(self):
        logging.info('Trying to get archiver executable')
        archiver = utils.sandbox_resource(resource_types.ARCHIVER_TOOL_EXECUTABLE, "build_dynamic_models", "75-4")

        logging.info('Get the first archive')
        arch1 = self.sync_resource(self.ctx[Archive1.name]) if self.ctx[Archive1.name] else None

        logging.info('Get the second archive')
        arch2 = self.sync_resource(self.ctx[Archive2.name]) if self.ctx[Archive2.name] else None

        logging.info('Calculate diff')
        diff = models_compare.compare_archives(archiver, arch1, arch2)
        self.ctx['diff'] = diff

        if self.ctx.get(StrictMode.name, StrictMode.default_value):
            changed = itertools.chain(diff.get('changed', []), diff.get('added', []), diff.get('deleted', []))
            production = sorted(p for p in changed if p.endswith(".info") and models_utils.is_production(p))
            eh.ensure(not production, 'Production models changed: {}'.format(production))


__Task__ = CompareDynamicModels
