# -*- coding: utf-8 -*-
import os

from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.base_search_quality import dolbilka_response_parser

import pool_comparator


class InputPool1(sp.ResourceSelector):
    name = 'estfeatures_pool_1'
    description = 'pool 1'
    resource_type = resource_types.IDX_OPS_EST_FEATURES_TSV_OUTPUT
    group = 'Task resources'


class InputPool2(sp.ResourceSelector):
    name = 'estfeatures_pool_2'
    description = 'pool 2'
    resource_type = resource_types.IDX_OPS_EST_FEATURES_TSV_OUTPUT
    group = 'Task resources'


class FactorNames(sp.ResourceSelector):
    name = 'factor_names'
    description = 'names of features'
    resource_type = resource_types.FACTOR_NAMES_TXT
    group = 'Task resources'


class CompareEstFeaturesOutputs(SandboxTask):
    """
        Сравнивает два выхлопa ``idx_ops`` в режиме ``estfeatures2f``.
    """
    type = 'COMPARE_ESTFEATURES_OUTPUTS'

    input_parameters = (
        InputPool1,
        InputPool2,
        FactorNames,
    )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self._create_resource(self.descr, 'compare_result', resource_types.ESTFEATURES_OUTPUTS_COMPARE_RESULT)
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        pool1 = self._read_resource(self.ctx[InputPool1.name])
        pool2 = self._read_resource(self.ctx[InputPool2.name])
        factorNames = self._read_resource(self.ctx[FactorNames.name])

        pool1FileName = pool1.abs_path()
        pool2FileName = pool2.abs_path()
        factorNamesFileName = factorNames.abs_path()

        resource = self._read_resource(self.ctx['out_resource_id'], sync=False)
        make_folder(resource.abs_path())

        parsedPool1 = dolbilka_response_parser.parse_and_group_by_query(pool1FileName, False, is_est_features=True)
        parsedPool2 = dolbilka_response_parser.parse_and_group_by_query(pool2FileName, False, is_est_features=True)
        diffs = pool_comparator.CompareResults(parsedPool1, parsedPool2)

        if not diffs.HasDifferences():
            noDifferencesFile = open(os.path.join(resource.abs_path(), 'noDifferences.txt'), 'w')
            print >>noDifferencesFile, 'there is no any differences between pools'
            noDifferencesFile.close()

            changed_factors = []
        else:
            FACTOR_NAMES = open(factorNamesFileName).read().split('\n')

            USER_REQUESTS = ["qid " + q for q in parsedPool1.keys()]

            pool_comparator.WriteDiffs(diffs, FACTOR_NAMES, USER_REQUESTS, resource.abs_path())

            differentFactors = diffs.GetDifferentFactors()

            changed_factors = []

            for factorNumber in sorted(differentFactors.keys()):
                changed_factors.append(
                    "FACTOR_%d (%s)" % (factorNumber, pool_comparator.GetFactorName(FACTOR_NAMES, factorNumber))
                )

            if diffs.HasNewFoundOrLostDocuments():
                changed_factors.append("DOCUMENTS")

        self.ctx['changed_factors'] = changed_factors

        size1 = os.path.getsize(pool1FileName)
        size2 = os.path.getsize(pool2FileName)

        if size1 == 0 or size2 == 0:
            raise SandboxTaskFailureError("incorrect file size")

        self.ctx['output_diff'] = 1.0 * size1 / size2

        resource.mark_ready()

    def get_short_task_result(self):
        if self.is_completed():
            if len(self.ctx['changed_factors']) > 0:
                return "diff"
            else:
                return "no diff"
        return None


__Task__ = CompareEstFeaturesOutputs
