# -*- coding: utf-8 -*-

import re
import difflib

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types


class MiddleOutput1ResourceParameter(parameters.LastReleasedResource):
    name = 'middle_output1_resource_id'
    description = 'Middle output 1'
    resource_type = resource_types.GEOMETASEARCH_OUTPUT


class MiddleOutput2ResourceParameter(parameters.LastReleasedResource):
    name = 'middle_output2_resource_id'
    description = 'Middle output 2'
    resource_type = resource_types.GEOMETASEARCH_OUTPUT


class UpperOutput1ResourceParameter(parameters.LastReleasedResource):
    name = 'upper_output1_resource_id'
    description = 'Upper output 1'
    resource_type = resource_types.GEOMETASEARCH_OUTPUT


class UpperOutput2ResourceParameter(parameters.LastReleasedResource):
    name = 'upper_output2_resource_id'
    description = 'Upper output 2'
    resource_type = resource_types.GEOMETASEARCH_OUTPUT


class CompareGeosearchOutputs(SandboxTask):
    type = 'COMPARE_GEOSEARCH_OUTPUTS'

    input_parameters = (
        MiddleOutput1ResourceParameter,
        MiddleOutput2ResourceParameter,
        UpperOutput1ResourceParameter,
        UpperOutput2ResourceParameter,
    )

    def on_execute(self):
        # just make almost same work twice
        params = (
            (self.ctx[UpperOutput1ResourceParameter.name], self.ctx[UpperOutput2ResourceParameter.name], 'upper'),
            (self.ctx[MiddleOutput1ResourceParameter.name], self.ctx[MiddleOutput2ResourceParameter.name], 'middle'),

        )

        for res1_id, res2_id, desc in params:
            def sync_and_get_resource(resource_id):
                self.sync_resource(resource_id)
                return channel.sandbox.get_resource(resource_id)
            res1 = sync_and_get_resource(res1_id)
            res2 = sync_and_get_resource(res2_id)

            res_out = self.create_resource(
                    'compare %s outputs' % desc,
                    '%s_diff.html' % desc,
                    'OTHER_RESOURCE'
                )
            self.ctx['%s_diff_resource_id' % desc] = res_out.id

            text1 = self.cut(open(res1.path).read(), desc)
            text2 = self.cut(open(res2.path).read(), desc)

            have_diff = self.write_diff(res_out.path, text1, text2)
            self.ctx['have_%s_diff' % desc] = have_diff

    def cut(self, text, desc):
        if desc == 'middle':
            return self.middle_cut(text)

        if desc == 'upper':
            return self.upper_cut(text)

        assert False

    def _cut(self, text, cuts):
        for re_, sub in cuts:
            text = re.sub(re_, sub, text)
        return text

    COMMON_CUTS = [
        (r'\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}.\d+Z', '0000-00-00T00:00:00Z'),
        (r'tasks/\d+/\d+/\d+/', 'tasks////'),
        (r' (.*/)*.+/.+:\d+', '<removed filename and line, see original output>'),
        (r'.cfg_\d+', '.cfg_<NN>'),  # Remove cfg suffix style1
        (r'_\d+.cfg', '_0.cfg'),  # Remove cfg suffix style2
        (r'ruid=\d+', 'ruid=0')  # Remove ruid, it is unstable
    ]

    def upper_cut(self, text):
        cuts = self.COMMON_CUTS
        return self._cut(text, cuts)

    def middle_cut(self, text):
        cuts = self.COMMON_CUTS + [
            (r'(request: /wizard\?)(.*)', r'request: /wizard\?<request cutted for stable diff, see it in original output>'),
        ]
        return self._cut(text, cuts)

    def write_diff(self, diff_path, text1, text2):
        differ = difflib.HtmlDiff()
        difftext = differ.make_file(text1.splitlines(), text2.splitlines(), context=True, numlines=3)

        with open(diff_path, 'w') as f:
            f.write(difftext)

        return text1 != text2


__Task__ = CompareGeosearchOutputs
