# -*- coding: utf-8 -*-

import copy
import os
import jinja2

from sandbox.sandboxsdk.paths import make_folder, copy_path
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types


class QueriesResourceParameter(parameters.LastReleasedResource):
    name = 'queries_resource_id'
    description = 'Queries'
    resource_type = resource_types.PLAIN_TEXT_QUERIES


class Responses1ResourceParameter(parameters.LastReleasedResource):
    name = 'respones1_resource_id'
    description = 'Responses 1'
    resource_type = resource_types.BASESEARCH_HR_RESPONSES


class ReqAnsLogs1ResourceParameter(parameters.LastReleasedResource):
    name = 'log1_resource_id'
    description = 'Logs 1'
    resource_type = resource_types.GEOMETASEARCH_REQANS_LOG


class Responses2ResourceParameter(parameters.LastReleasedResource):
    name = 'respones2_resource_id'
    description = 'Responses 2'
    resource_type = resource_types.BASESEARCH_HR_RESPONSES


class ReqAnsLogs2ResourceParameter(parameters.LastReleasedResource):
    name = 'log2_resource_id'
    description = 'Logs 2'
    resource_type = resource_types.GEOMETASEARCH_REQANS_LOG


class MiddleOut1ResourceParameter(parameters.LastReleasedResource):
    name = 'middle_out1_resource_id'
    description = 'Middle out 1'
    resource_type = resource_types.GEOMETASEARCH_OUTPUT
    required = False


class MiddleOut2ResourceParameter(parameters.LastReleasedResource):
    name = 'middle_out2_resource_id'
    description = 'Middle out 2'
    resource_type = resource_types.GEOMETASEARCH_OUTPUT
    required = False


class UpperOut1ResourceParameter(parameters.LastReleasedResource):
    name = 'upper_out1_resource_id'
    description = 'Upper out 1'
    resource_type = resource_types.GEOMETASEARCH_OUTPUT
    required = False


class UpperOut2ResourceParameter(parameters.LastReleasedResource):
    name = 'upper_out2_resource_id'
    description = 'Upper out 2'
    resource_type = resource_types.GEOMETASEARCH_OUTPUT
    required = False


class CompareGeosearchResponses(SandboxTask):
    type = 'COMPARE_GEOSEARCH_RESPONSES'

    input_parameters = (
        QueriesResourceParameter,
        Responses1ResourceParameter,
        ReqAnsLogs1ResourceParameter,
        Responses2ResourceParameter,
        ReqAnsLogs2ResourceParameter,
        MiddleOut1ResourceParameter,
        MiddleOut2ResourceParameter,
        UpperOut1ResourceParameter,
        UpperOut2ResourceParameter,
    )

    def on_execute(self):
        if 'subtasks_created' not in self.ctx:
            compare_responses_task, compare_logs_task, outs_task = self.start_subtasks()

            self.ctx['compare_responses_task_id'] = compare_responses_task.id
            self.ctx['compare_logs_task_id'] = compare_logs_task.id
            if outs_task:
                self.ctx['compare_outputs_task_id'] = outs_task.id
            self.ctx['subtasks_created'] = True

            self.wait_all_tasks_stop_executing([compare_responses_task.id, compare_logs_task.id])
        else:
            self.process_subtask_results()

    def start_subtasks(self):
        queries_resource = channel.sandbox.get_resource(self.ctx[QueriesResourceParameter.name])
        respones1_resource = channel.sandbox.get_resource(self.ctx[Responses1ResourceParameter.name])
        respones2_resource = channel.sandbox.get_resource(self.ctx[Responses2ResourceParameter.name])

        responses_task = self.create_subtask(
            task_type='COMPARE_BASESEARCH_RESPONSES',
            description=self.descr + ': compare responses',
            input_parameters={
                'queries_resource_id': queries_resource.id,
                'basesearch_responses1_resource_id': respones1_resource.id,
                'basesearch_responses2_resource_id': respones2_resource.id,
                'save_simple_diff': False,
                'remove_index_generation': True,
                'queries_per_file': 100,
                'response_format': 'geosearch',
            },
        )

        log1_resource = channel.sandbox.get_resource(self.ctx[ReqAnsLogs1ResourceParameter.name])
        log2_resource = channel.sandbox.get_resource(self.ctx[ReqAnsLogs2ResourceParameter.name])
        logs_task = self.create_subtask(
            task_type='COMPARE_GEOSEARCH_REQANS_LOGS',
            description=self.descr + ': compare reqans logs',
            input_parameters={
                'queries_resource_id': queries_resource.id,
                'log1_resource_id': log1_resource.id,
                'log2_resource_id': log2_resource.id,
            },
        )

        outs_task = None
        # if self.ctx.get(MiddleOut1ResourceParameter.name):
        #    mout1_resource = channel.sandbox.get_resource(self.ctx[MiddleOut1ResourceParameter.name])
        #    mout2_resource = channel.sandbox.get_resource(self.ctx[MiddleOut2ResourceParameter.name])
        #    uout1_resource = channel.sandbox.get_resource(self.ctx[UpperOut1ResourceParameter.name])
        #    uout2_resource = channel.sandbox.get_resource(self.ctx[UpperOut2ResourceParameter.name])

        #    outs_task = self.create_subtask(
        #        task_type='COMPARE_GEOSEARCH_OUTPUTS',
        #        description=self.descr + ': compare outputs',
        #        input_parameters={
        #            'middle_output1_resource_id': mout1_resource.id,
        #            'middle_output2_resource_id': mout2_resource.id,
        #            'upper_output1_resource_id': uout1_resource.id,
        #            'upper_output2_resource_id': uout2_resource.id,
        #        },
        #    )

        return responses_task, logs_task, outs_task

    def _copy_resource(self, resource_id, dst_path):
        resource = channel.sandbox.get_resource(resource_id)
        src_path = self.sync_resource(resource.id)
        # dst_path = resource.file_name
        copy_path(src_path, dst_path)

        attributes = copy.copy(resource.attributes)
        attributes.pop('released', None)
        created_resource = self.create_resource(resource.description, dst_path, str(resource.type), arch=resource.arch,
                                                attributes=attributes)
        self.mark_resource_ready(created_resource)
        return created_resource

    def process_subtask_results(self):
        responses_task = channel.sandbox.get_task(self.ctx['compare_responses_task_id'])
        if responses_task.new_status != self.Status.SUCCESS:
            raise Exception('Compare responses task not successful')

        logs_task = channel.sandbox.get_task(self.ctx['compare_logs_task_id'])
        if logs_task.new_status != self.Status.SUCCESS:
            raise Exception('Compare reqans logs task not successful')

        # outs_task = None
        # if self.ctx.get('compare_outputs_task_id'):
        #    outs_task = channel.sandbox.get_task(self.ctx['compare_outputs_task_id'])
        #    if outs_task.new_status != self.Status.SUCCESS:
        #        raise Exception('Compare outputs task not successful')

        diff_in_responses = not responses_task.ctx['compare_result']
        diff_in_logs = logs_task.ctx['diff_count'] != 0

        diff_in_middle_outs, diff_in_upper_outs = False, False
        # if outs_task:
        #    diff_in_middle_outs = outs_task.ctx['have_middle_diff']
        #    diff_in_upper_outs = outs_task.ctx['have_upper_diff']
        self.ctx['have_diff'] = any([diff_in_responses, diff_in_logs, diff_in_middle_outs, diff_in_upper_outs])

        responses_diff_url = None
        if diff_in_responses:
            responses_diff_url = self._copy_resource(responses_task.ctx['out_resource_id'], "responses_diff").proxy_url

        logs_diff_url = None
        if diff_in_logs:
            logs_diff_url = self._copy_resource(logs_task.ctx['out_resource_id'], 'logs_diff').proxy_url
            logs_diff_url += '/index.html'

        middle_outs_diff_url = None
        # if diff_in_middle_outs:
        #    middle_outs_diff_url = channel.sandbox.get_resource(outs_task.ctx['middle_diff_resource_id']).url

        upper_outs_diff_url = None
        # if diff_in_upper_outs:
        #    upper_outs_diff_url = channel.sandbox.get_resource(outs_task.ctx['upper_diff_resource_id']).url

        logs_task = channel.sandbox.get_task(self.ctx['compare_logs_task_id'])

        template = get_template()

        text = template.render(
            diff_in_responses=diff_in_responses,
            responses_diff_url=responses_diff_url,
            diff_in_logs=diff_in_logs,
            logs_diff_url=logs_diff_url,
            diff_in_middle_outs=diff_in_middle_outs,
            middle_outs_diff_url=middle_outs_diff_url,
            diff_in_upper_outs=diff_in_upper_outs,
            upper_outs_diff_url=upper_outs_diff_url,
        )

        out_resource = self.create_resource(
            'compare result',
            'compare_result',
            'OTHER_RESOURCE'

        )
        self.ctx['out_resource_id'] = out_resource.id
        make_folder(out_resource.path)

        file_name = os.path.join(out_resource.path, 'index.html')

        with open(file_name, 'w') as f:
            f.write(text.encode('utf-8'))


def get_template():
    template_path = os.path.dirname(os.path.abspath(__file__))
    env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
    return env.get_template('report.jinja2')


__Task__ = CompareGeosearchResponses
