# -*- coding: utf-8 -*-

import logging
import traceback

from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel


class LoadLog1Resource(sp.ResourceSelector):
    name = 'load_log1_resource_id'
    description = '1st LoadLog'
    resource_type = 'LOAD_LOG'
    required = True


class LoadLog2Resource(sp.ResourceSelector):
    name = 'load_log2_resource_id'
    description = '2nd LoadLog'
    resource_type = 'LOAD_LOG'
    required = True


class CompareLoadlogs(SandboxTask):
    """
        Анализирует два LoadLog-а, и сравнивает нагрузочные параметры (cpu load, etc)
    """
    type = 'COMPARE_LOADLOGS'

    input_parameters = [
        LoadLog1Resource,
        LoadLog2Resource,
    ]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self._create_resource(self.descr, 'compare_result.html', resource_types.LOAD_LOG_COMPARE_RESULT)
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        load_log1_resource_path = self.sync_resource(self.ctx['load_log1_resource_id'])
        load_log2_resource_path = self.sync_resource(self.ctx['load_log2_resource_id'])

        analyze_result1 = _parse_loadlog(load_log1_resource_path)
        analyze_result2 = _parse_loadlog(load_log2_resource_path)

        resource_path = channel.sandbox.get_resource(self.ctx['out_resource_id']).path

        self.ctx['has_diff'] = _save_diff(resource_path, analyze_result1, analyze_result2)

        self.mark_resource_ready(self.ctx['out_resource_id'])


__Task__ = CompareLoadlogs


def _parse_loadlog(fileName):
    nLines = 0

    sumSearchTime = 0
    foundDocCount = 0
    sumCpuTime = 0
    noDocFoundErrCount = 0
    timeoutErrCount = 0
    otherErr = {}
    sumInQueryAndSearchTime = 0

    for line in open(fileName).read().split('\n'):
        if not line:
            continue

        try:
            line = line.split("\t")
            if len(line) == 4:
                continue

            nLines += 1

            sumSearchTime += int(line[0])

            foundDocCount += int(line[5])

            sumCpuTime += int(line[8])

            err = line[9]
            if err != "0":
                if err == "15":
                    noDocFoundErrCount += 1
                elif err == "20":
                    timeoutErrCount += 1
                else:
                    otherErr[err] = otherErr.get(err, 0) + 1

            sumInQueryAndSearchTime += int(line[10])
        except Exception as error:
            raise Exception("cannot parse line '%s'\nexception type: %s\n\nerror:\n%s" % (line, type(error), traceback.format_exc()))

    return {
        "avg_search_time": sumSearchTime / nLines,
        "avg_found_doc_count": foundDocCount / nLines,
        "avg_cpu_time": sumCpuTime / nLines,
        "no_doc_found_err_count": noDocFoundErrCount,
        "timeout_err_count": timeoutErrCount,
        "other_err": otherErr,
        "avg_in_query_and_search_time": sumInQueryAndSearchTime / nLines
    }


class SimpleComparer:
    def __init__(self, diff):
        self.Diff = diff

    def Compare(self, file, key, value1, value2):
        hasDiff = False

        if value1 != value2:
            if value1 == 0 or value2 == 0:
                hasDiff = True
                diff = "<-"
            else:
                diff = float(max(value1, value2)) / min(value1, value2)
                if diff > self.Diff:
                    hasDiff = True

                if value1 > value2:
                    diff = "&darr;%10.3f" % diff
                else:
                    diff = "&uarr;%10.3f" % diff
        else:
            diff = "no diff"

        _write_row(file, key, value1, value2, diff, color="red" if hasDiff else None)

        return hasDiff


class TimeoutErrComparer(SimpleComparer):
    # Timeouts count is unstable
    def Compare(self, file, key, value1, value2):
        if value1 < 300 and value2 < 300:
            _write_row(file, key, value1, value2, "no diff", None)
            return False

        return SimpleComparer.Compare(self, file, key, value1, value2)


class ErrComparer:
    def Compare(self, file, key, errors1, errors2):
        hasDiff = False

        if errors1 != errors2:
            hasDiff = True
            diff = "diff"
        else:
            diff = "no diff"

        _write_row(file, key, errors1, errors2, diff, color="red" if hasDiff else None)

        return hasDiff


_COMPARERS = {
    "avg_search_time": SimpleComparer(2),
    "avg_found_doc_count": SimpleComparer(1.2),
    "avg_cpu_time": SimpleComparer(1.3),
    "no_doc_found_err_count": SimpleComparer(1.2),
    "timeout_err_count": TimeoutErrComparer(1.5),
    "other_err": ErrComparer(),
    "avg_in_query_and_search_time": SimpleComparer(2)
}


def _save_diff(fileName, analyze_result1, analyze_result2):
    hasDiff = False

    with open(fileName, "w") as diffFile:
        diffFile.write('<html><body>')
        diffFile.write('<table style="border: 1px solid #afafaf; border-collapse: collapse;">\n')
        _write_row(diffFile, "property", "data1", "data2", "diff (max value / min value)")

        for key, data1 in analyze_result1.iteritems():
            data2 = analyze_result2[key]

            logging.info("key: '%s', data1: '%s', data2: '%s'", key, data1, data2)

            if _compare_single_key(diffFile, key, data1, data2):
                hasDiff = True

        diffFile.write("</table></body></html>")

    return hasDiff


def _write_cell(f, value):
    f.write('  <td style="border: 1px solid #afafaf">{}</td>\n'.format(value))


def _write_row(f, v1, v2, v3, v4, color=None):
    if not color:
        color = "black"
    f.write('<tr style="color: {}">\n'.format(color))
    _write_cell(f, v1)
    _write_cell(f, v2)
    _write_cell(f, v3)
    _write_cell(f, v4)
    f.write('</tr>\n')


def _compare_single_key(file, key, data1, data2):
    hasDiff = False

    comparer = _COMPARERS[key]
    if comparer.Compare(file, key, data1, data2):
        hasDiff = True

    return hasDiff
