import logging
import os
from os.path import join as pj, basename, exists as pexists
import hashlib
import re
import subprocess
import json

from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import (
    SandboxBoolParameter,
    SandboxStringParameter,
    ResourceSelector,

)
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import get_unique_file_name, make_folder
import sandbox.sandboxsdk.util as sdk_util

from sandbox.projects import resource_types

from sandbox.projects.common.binaries_provider_ymake import BinariesProvider, RequiredBinary, OptionalBinary
from sandbox.projects.common.ProcessPool import ProcessPool
from sandbox.projects.common.mapreduce_stored_tables import SortMode
from sandbox.projects.common.userdata import mr_base_task, util

import pprint
pp = pprint.PrettyPrinter(indent=2).pprint


def binary_version(ctx, ctxname, execname):
    if ctxname not in ctx:
        ctx[ctxname] = 0
        for key in '-version', '--version':
            if execname in ctx:
                p = run_process([ctx[execname], key], shell=False, wait=False, stdout=subprocess.PIPE)
                (hlp, _) = p.communicate()
                m = re.search(r'Last Changed Rev: ([0-9]+)', hlp)
                if m:
                    ctx[ctxname] = int(m.group(1))
                    break

    logging.info("Got version for " + execname + ": " + str(ctx[ctxname]))
    return ctx[ctxname]


class StateAPrefixParameter(SandboxStringParameter):
    """
        Table name prefix of source state
    """
    name = 'state_a_prefix'
    description = "(A) table name prefix of state:"
    required = False
    default_value = 'userdata/'
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


class StateAResourceParameter(ResourceSelector):
    """
        Get MR data (A) from resource
    """
    name = 'state_a_resource_id'
    description = "(A) resource with stored tables:"
    required = False
    resource_type = resource_types.USERDATA_TABLES_ARCHIVE
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


class StateAObtainSelector(SandboxStringParameter):
    """
        Source of state (A) tables
    """
    name = 'state_a_source_selector'
    description = '(A) source of state tables:'
    choices = [('MR', 'MR'), ('resource', 'resource')]
    sub_fields = {'MR': [StateAPrefixParameter.name], 'resource': [StateAResourceParameter.name]}
    default_value = 'resource'
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


class StateBPrefixParameter(SandboxStringParameter):
    """
        Table name prefix of source state
    """
    name = 'state_b_prefix'
    description = "(B) table name prefix of state:"
    required = False
    default_value = 'userdata/'
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


class StateBResourceParameter(ResourceSelector):
    """
        Get MR data (B) from resource
    """
    name = 'state_b_resource_id'
    description = "(B) resource with stored tables:"
    required = False
    resource_type = resource_types.USERDATA_TABLES_ARCHIVE
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


class StateBObtainSelector(SandboxStringParameter):
    """
        Source of state (B) tables
    """
    name = 'state_b_source_selector'
    description = '(B) source of state tables:'
    choices = [('MR', 'MR'), ('resource', 'resource')]
    sub_fields = {'MR': [StateBPrefixParameter.name], 'resource': [StateBResourceParameter.name]}
    default_value = 'resource'
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


class DumpTablesToTextParameter(SandboxBoolParameter):
    name = 'dump_to_text'
    description = "Dump eligible tables to json or text:"
    default_value = False
    group = mr_base_task.OUTPUT_PARAMS_GROUP_NAME


class ColoredDiffParameter(SandboxBoolParameter):
    name = 'colored_diff'
    description = "Colored side-by-side diff in HTML for dumpable protobuf tables:"
    default_value = False
    group = mr_base_task.OUTPUT_PARAMS_GROUP_NAME


class StoreTablesParameter(SandboxBoolParameter):
    name = 'store_tables'
    description = "Store resulting diff as sandbox resource:"
    default_value = True
    group = mr_base_task.OUTPUT_PARAMS_GROUP_NAME


class TablesToSkipParameter(SandboxStringParameter):
    name = 'skip_tables'
    description = "Don't diff these tables:"
    required = False
    multiline = True
    group = mr_base_task.OUTPUT_PARAMS_GROUP_NAME


class FragmentAResourceParameter(ResourceSelector):
    name = 'fragment_a_resource_id'
    description = "(A) resource with dump:"
    required = False
    resource_type = resource_types.USERDATA_INDEX_FRAGMENT
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


class FragmentBResourceParameter(ResourceSelector):
    name = 'fragment_b_resource_id'
    description = "(B) resource with dump:"
    required = False
    resource_type = resource_types.USERDATA_INDEX_FRAGMENT
    group = mr_base_task.INPUT_PARAMS_GROUP_NAME


def hashfile(afile):
    hasher = hashlib.md5()
    blocksize = 1024 * 1024
    buf = afile.read(blocksize)
    while len(buf) > 0:
        hasher.update(buf)
        buf = afile.read(blocksize)
    return hasher.hexdigest()


class CompareMRTables(mr_base_task.Task):
    """
        Compare two MR 'directories' recursively. Each 'directory' can be supplied as either real
        MR path or stored as resource.
    """

    type = "COMPARE_MR_TABLES"
    execution_space = 100000
    # noinspection PyUnresolvedReferences
    client_tags = (Tag.GENERIC | Tag.CUSTOM_USER_DATA) & mr_base_task.Task.client_tags
    cores = 17

    need_rem = False
    store_resulting_tables = False
    sort_resulting_tables = False
    yt_testable = False

    binaries = (
        [
            RequiredBinary("quality/mr_apps/mr_diff", "mr_diff"),
            OptionalBinary("yweb/robot/userdata/userdata_view", "userdata_view"),
            OptionalBinary("yweb/querydata/querydata_viewer", "querydata_viewer")
        ] +
        mr_base_task.Task.binaries
    )

    binaries_provider = BinariesProvider(binaries)

    input_parameters = util.smart_join_params(
        mr_base_task.Task.input_parameters,
        StateAObtainSelector,
        StateAPrefixParameter,
        StateAResourceParameter,
        StateBObtainSelector,
        StateBPrefixParameter,
        StateBResourceParameter,
        FragmentAResourceParameter,
        FragmentBResourceParameter,
        DumpTablesToTextParameter,
        ColoredDiffParameter,
        StoreTablesParameter,
        TablesToSkipParameter,
        *binaries_provider.task_parameters()
    )

    def process_mr_data(self):
        self.ctx['has_diff'] = False  # is set in compare_dumps or here

        self.compare_dumps()
        diff_tables = self.compare_tables()

        if diff_tables:
            self.ctx['has_diff'] = True
            self.ctx['diff_tables'] = diff_tables
            if self.ctx.get(StoreTablesParameter.name):
                self.store_resulting_tables = True
                logging.info(pprint.pformat(diff_tables))
            if self.ctx.get(DumpTablesToTextParameter.name):
                self.dump_diff_tables(diff_tables)

    def userdata_view_version(self):
        return binary_version(self.ctx, 'userdata_view_version', 'userdata_view_executable')

    def can_userdata_view(self, filename):
        viewer = self.ctx.get('userdata_view_executable')
        if viewer is None:
            return False
        p = run_process([viewer, filename, "--help"], shell=False, wait=False)
        return p.wait() == 0

    def index_metadata_for(self, base):
        k = base + '_meta_file'
        if k not in self.ctx:
            for root, _, files in os.walk(base):
                if 'fasttier' not in root and 'meta' in files:
                    self.ctx[k] = pj(root, 'meta')
                    break

        return self.ctx[k]

    def view_indexuserdat_new(self, base, infile, outfile):
        return "{bin} {infile} --meta {metadata} > {outfile}".format(
            bin=self.ctx['userdata_view_executable'],
            infile=infile,
            outfile=outfile,
            metadata=self.index_metadata_for(base)
        )

    def viewer_for_file(self, relname, thisdir):  # should return either formattable string with {infile} and {outfile} placeholders, or a callback with those two as named parameters
        can_ud_view = self.can_userdata_view(relname)
        if can_ud_view and re.search("/indexuser(url|own|q2)(.[0-9]+)?.dat.gz", relname):
            return lambda infile, outfile: self.view_indexuserdat_new(thisdir, infile, outfile)

        if self.ctx.get("querydata_viewer_executable"):
            if re.search(r'similar_docs(_v2)?.trie.[0-9]+', relname):
                return self.ctx['querydata_viewer_executable'] + ' -H -i {infile} > {outfile}'

            if re.search(r'indexuserown.qd/[0-9]+/indexuserown.[0-9]+', relname):
                return self.ctx['querydata_viewer_executable'] + ' -H -i {infile} > {outfile}'

        if re.search(r'\.d2c$|\.c2n$|\.c2p$|\.txt$|/meta$|indexuserown.tag$|/results.*(stats|markers|observables)$', relname):
            return 'cp {infile} {outfile}'

        if can_ud_view:
            return self.ctx['userdata_view_executable'] + ' {infile} > {outfile}'

        return None

    def skip_file_comparision(self, relname, thisdir, thatdir):
        if relname.endswith('indexuserown.qd.tar'):
            return True
        if relname.endswith("/results.sizes"):  # skip comparision of unstable USERFEAT-470
            return True
        if 'indexoq.all' in relname:
            return True
        if relname.endswith('inv'):
            keyfile = relname[:-3] + 'key'
            if pexists(pj(thisdir, keyfile)) and pexists(pj(thatdir, keyfile)):
                logging.info("XXX: skipping inv file " + relname + " because printkeys will dump both")
                return True

    def get_md5_diff(self, first_name, second_name):
        first_md5 = hashfile(open(first_name))
        second_md5 = hashfile(open(second_name))
        if first_md5 != second_md5:
            logging.info("Files {} and {} have different md5: {} and {}".format(first_name, second_name, first_md5, second_md5))
            return first_md5, second_md5
        else:
            logging.info("Files {} and {} have same md5: {}".format(first_name, second_name, first_md5))
            return None

    def compare_dumps(self):
        parts = {'a': {}, 'b': {}}
        for (st, d) in parts.items():
            resid = self.ctx['fragment_' + st + '_resource_id']
            if resid is None or resid == 0:
                return  # not specified, assume we don't need to compare files
            d['resource'] = channel.sandbox.get_resource(resid)
            d['folder'] = self.sync_resource(resid)
            d['files'] = {}
            for r, _, ff in os.walk(d['folder']):
                for f in ff:
                    fn = pj(r, f)
                    if os.path.islink(fn):
                        continue  # for some reason 'create_resource' preserves symlinks that point outside of the tree. Ignore them for now
                    rfn = os.path.relpath(fn, d['folder'])
                    d['files'][rfn] = os.stat(fn)
        diff = {
            'size': {},
            'only_in_a': [],
            'only_in_b': [],
            'content': {}
        }
        seen_diff = set()
        diff_folder = get_unique_file_name(self.abs_path(''), 'diff_files')
        make_folder(diff_folder)
        for n_this, n_that in (('a', 'b'), ('b', 'a')):
            this = parts[n_this]['files']
            that = parts[n_that]['files']
            for file_name, file_stats in this.items():
                if file_name not in that:
                    diff.setdefault('only_in_' + n_this, []).append(file_name)
                    continue
                if file_name in seen_diff:
                    continue

                seen_diff.add(file_name)
                if self.skip_file_comparision(file_name, parts[n_this]['folder'], parts[n_that]['folder']):
                    continue

                md5_diff = self.get_md5_diff(pj(parts[n_this]['folder'], file_name), pj(parts[n_that]['folder'], file_name))
                if md5_diff is None:
                    continue

                this_viewer = self.viewer_for_file(file_name, parts[n_this]['folder'])
                that_viewer = self.viewer_for_file(file_name, parts[n_that]['folder'])
                if not (this_viewer and that_viewer):
                    if file_stats.st_size != that[file_name].st_size:
                        diff['size'][file_name] = {n_this: file_stats.st_size, n_that: that[file_name].st_size}
                    elif file_name not in diff['content']:
                        # already compared earlier by md5
                        logging.info("Comparing {a,b}/" + file_name + " via plain MD5: no viewer")
                        diff['content'][file_name] = {
                            n_this: md5_diff[0],
                            n_that: md5_diff[1]
                        }
                    continue

                safe_file_name = file_name.replace(os.sep, '.')
                dumps = {}
                for part, viewer in ((n_this, this_viewer), (n_that, that_viewer)):
                    dumps[part] = get_unique_file_name(self.abs_path(''), part + '_' + safe_file_name)
                    infile = pj(parts[part]['folder'], file_name)
                    outfile = dumps[part]
                    if callable(viewer):
                        cmd = viewer(infile, outfile)
                    else:
                        cmd = viewer.format(infile=infile, outfile=outfile, **self.ctx)

                    p = run_process(
                        cmd,
                        shell=True,
                        wait=True,
                        check=False,
                        log_prefix='userdata_view',
                    )
                    if p.returncode > 0:
                        with open(dumps[part], 'w') as d:
                            print >>d, "Error dumping ", pj(parts[part]['folder'], file_name)
                md5_dump_diff = self.get_md5_diff(dumps[n_this], dumps[n_that])
                if md5_dump_diff is not None:
                    diff_file = get_unique_file_name(diff_folder, safe_file_name + '.diff')
                    p = run_process(
                        '/usr/bin/diff -wbu -U 6 --speed-large-files {this_dump} '
                        '{that_dump} > {diff_file}'.format(
                            that_dump=dumps[n_that],
                            this_dump=dumps[n_this],
                            diff_file=pj(diff_folder, diff_file)
                        ),
                        shell=True,
                        wait=True,
                        check=False,
                        log_prefix='usr.bin.diff',
                    )
                    if p.returncode > 1:
                        raise SandboxTaskFailureError("Error executing diff")
                    diff['content'][file_name] = {
                        'dump_' + n_this: md5_dump_diff[0],
                        'dump_' + n_that: md5_dump_diff[1],
                        'diff': basename(diff_file)
                    }
                os.remove(dumps[n_this])
                os.remove(dumps[n_that])

        if any([bool(i) for i in diff.values()]):
            self.ctx['has_diff'] = True
            self.ctx['diff_text'] = diff
            outfile = pj(diff_folder, 'diff.txt')
            with open(outfile, 'w') as o:
                pprint.pprint(diff, o)
            resource = util.get_or_create_resource(
                self,
                self.ctx.get("diff_text_resource_id"),
                description="description of diff between userdata indexi {} and {}".format(
                    parts['a']['resource'].id,
                    parts['b']['resource'].id
                ),
                resource_path=diff_folder,
                resource_type=resource_types.USERDATA_INDEX_FRAGMENT_DIFF_DESCR,
                arch='any',
            )
            self.mark_resource_ready(resource.id)
            self.ctx["diff_text_resource_id"] = resource.id
            self.ctx["diff_text_resource_url"] = resource.proxy_url

    def check_params(self):
        if self.ctx['mr_server'].endswith("local"):
            for st in 'a', 'b':
                if (
                    self.ctx['state_' + st + '_source_selector'] == 'MR' or
                    self.ctx.get('state_' + st + '_resource_id') is None
                ):
                    raise SandboxTaskFailureError("Need source state '{}' to run locally".format(st))

    def get_prefix_for(self, suffix):
        return self.get_tables_prefix() + suffix + "/"

    def prepare_mr(self):
        a_n2d = {}
        b_n2d = {}
        for st, n2d in [('a', a_n2d), ('b', b_n2d)]:
            rid = self.ctx.get('state_' + st + '_resource_id')
            if rid is not None:
                self.ctx['state_' + st + '_prefix'] = self.get_prefix_for(st)
                to_upload = self.mr_tables_io.get_resource_table_descrs(
                    self, rid, new_prefix=self.get_prefix_for(st)
                )
                for tdesc in to_upload:
                    n2d[tdesc['shortname']] = tdesc

        for tname in set(a_n2d.keys() + b_n2d.keys()):
            ta = a_n2d.get(tname, {})
            tb = b_n2d.get(tname, {})
            if ta.get('sort_mode') != tb.get('sort_mode') or ta.get("sort_mode") == SortMode.NONE:
                logging.info("Not sorted or one side only: " + tname)
                continue

            p = run_process(
                "bash -c 'cmp -s <(zcat {a}) <(zcat {b})' 2>/dev/null".format(
                    a=ta["fullpath"],
                    b=tb["fullpath"]
                ),
                shell=True, wait=True, check=False,
                log_prefix="cmp_tables"
            )
            if p.returncode == 0:
                logging.info("table {} are same in both states, skipping".format(tname))
                a_n2d.pop(tname)
                b_n2d.pop(tname)
            p = None  # XXX seems like there's a leak of filedescriptors otherwise

        logging.info("nonequal or nonsorted: " + repr(a_n2d) + " " + repr(b_n2d))

        p = ProcessPool(max(1, sdk_util.system_info()['ncpu'] / 2))
        p.map(lambda tdesc: self.mr_tables_io.upload_table(tdesc), a_n2d.values() + b_n2d.values())

    def compare_tables(self):
        # intra-mr comparision
        a_list = self.mr_client.get_tables_list(self.ctx['state_a_prefix'])
        b_list = self.mr_client.get_tables_list(self.ctx['state_b_prefix'])
        a_set = set(map(lambda n: n[len(self.ctx['state_a_prefix']):], a_list))
        b_set = set(map(lambda n: n[len(self.ctx['state_b_prefix']):], b_list))
        if a_set != b_set:
            copy_tables = []
            for tname in a_set - b_set:
                copy_tables.append([
                    self.ctx['state_a_prefix'] + tname,
                    self.get_prefix_for("diff/a_only") + tname
                ])
            for tname in b_set - a_set:
                copy_tables.append([
                    self.ctx['state_b_prefix'] + tname,
                    self.get_prefix_for("diff/b_only") + tname
                ])
            p = ProcessPool(max(1, sdk_util.system_info()['ncpu'] / 2))
            p.map(lambda (src, dst): self.mr_client.copy_table(src, dst), copy_tables)

        skips = set(self.ctx['skip_tables'].splitlines()) if self.ctx.get('skip_tables') else set([])
        logging.info(pprint.pformat(skips))

        pr = util.ProcessRunner()
        for tname in a_set.intersection(b_set):
            is_yamr_compatible_format = all(
                self.mr_client.is_yamr_compatible_format(prefix + tname)
                for prefix in (self.ctx['state_a_prefix'], self.ctx['state_b_prefix'])
            )
            if not is_yamr_compatible_format:
                logging.info('Skipping because of non-YAMR compatible format: table ' + tname)
                continue

            if tname in skips or tname.endswith('/filtered/web/oqt_data') or tname.endswith('/filtered/web/production/oqt_data'):
                logging.info('Skipping table ' + tname)
                continue
            tskv = ''
            if self.table_is_tskv(self.ctx['state_a_prefix'] + tname) and self.table_is_tskv(self.ctx['state_b_prefix'] + tname):
                tskv = ' -tskv -context -tskv-add-key reg -max-dups 1000 '
            pr.add(
                "mr_diff",
                '{env} {mr_diff_executable} -s {real_mr_server} {tskv} '
                '{state_a_prefix}{tname} {state_b_prefix}{tname} {tables_prefix}{tname}',
                env=self.mr_client.get_env_str(),
                tname=tname, tskv=tskv, tables_prefix=self.get_prefix_for("diff"), **self.ctx
            )
        pr.run()
        return self.mr_client.get_tables_list(self.get_prefix_for("diff"))

    def get_table_head(self, tn):
        recs_specifier = "[:#10]" if self.ctx["mr_server"] == "yt_local" else ":[0,10]"  # XXX: adapt to real MR
        p = self.mr_client.run(
            '-read {table}{rs} -subkey'.format(table=tn, rs=recs_specifier),
            check=True,
            wait=False,
            outputs_to_one_file=False,
            stdout=subprocess.PIPE,
            log_prefix="mapreduce-get-head"
        )
        (head, _) = p.communicate()
        return head

    def head_is_text(self, head, tn='<unk>'):
        if not head:  # empty or some silent error while reading
            return False
        for l in head.splitlines():
            try:
                l.decode('utf-8')
                if l != l.translate(None, '\000\001\002\003\004\005\006\007\010\013\014\016\017\020\021\022\023\024\025\026\027\030\031\032\033\034\035\036\037'):
                    # only allow tabs, CRs, LFs
                    return False
            except (UnicodeDecodeError, UnicodeEncodeError):
                logging.info('table ' + tn + ' contains non-utf lines')
                return False
        logging.info("Table {} contains text data".format(tn))
        return True

    def head_is_tskv(self, head, tn='<unk>'):
        if not head:  # empty or some silent error while reading
            return False
        # please make sure you don't call this with binary data
        for l in head.splitlines():
            f = l.split('\t')
            if len(f) < 3:  # XXX can't happen, it's k, sk, v
                return False
            if len(f[-1].split('=')) != 2:
                return False
        logging.info("Table {} contains tskv data".format(tn))
        return True

    def table_is_tskv(self, tn):
        head = self.get_table_head(tn)
        return self.head_is_text(head, tn) and self.head_is_tskv(head, tn)

    def table_is_text(self, tn):
        head = self.get_table_head(tn)
        return self.head_is_text(head, tn)

    def dump_diff_tables(self, diff_tables):
        diff_folder = get_unique_file_name(self.abs_path(''), 'diff_tables_dump')
        make_folder(diff_folder)
        dumped = False
        pr = util.ProcessRunner()
        table_to_textfile = {}
        for tn in diff_tables:
            file_basename = tn.replace('/', ':')
            filename_prefix = pj(diff_folder, file_basename)
            if self.can_userdata_view(tn):
                raw_dump = "a_only/" in tn or "b_only/" in tn
                colored_diff = "--colored_diff" if self.ctx.get('colored_diff') else ""
                if colored_diff and raw_dump:
                    logging.warning("colored_diff was disabled because raw_dump is in place")
                    colored_diff = False
                filename_extension = ".html" if colored_diff else ".txt"
                cmd = '{env} {userdata_view} {table} {mr_diff_format} {colored_diff} --server {real_mr_server} > {filename_prefix}{filename_extension}'.format(
                    real_mr_server=self.ctx['real_mr_server'],
                    env=self.mr_client.get_env_str(),
                    userdata_view=self.ctx['userdata_view_executable'],
                    table=tn,
                    filename_prefix=filename_prefix,
                    filename_extension=filename_extension,
                    mr_diff_format="" if raw_dump else "--mr_diff_format",
                    colored_diff=colored_diff,
                )
                process = run_process(
                    cmd,
                    shell=True,
                    check=False,
                    wait=True,
                    log_prefix='userdata_view.table.' + file_basename + filename_extension,
                )
                if not process.returncode:
                    table_to_textfile[tn] = file_basename + filename_extension
                    dumped = True
                    continue
                logging.warning("Command '{}' died with exit code {}".format(cmd, process.returncode))
            if self.table_is_text(tn):
                filename_extension = ".txt"
                table_to_textfile[tn] = file_basename + filename_extension
                dumped = True
                pr.add("mapreduce-read-dump-txt", self.mr_client.command(
                    '-subkey -read {table} > {filename_prefix}{filename_extension}'.format(
                        table=tn,
                        filename_prefix=filename_prefix,
                        filename_extension=filename_extension,
                    )
                ))
            else:
                """
                Let's dump as YSON when we don't have a better alternative.
                """
                filename_extension = ".yson"
                table_to_textfile[tn] = file_basename + filename_extension
                dumped = True
                pr.add("dump-diff-table-from-yt-as-yson", self.mr_client.command(
                    "-format '<format=pretty>yson' -read {table} > {filename_prefix}{filename_extension}".format(
                        table=tn,
                        filename_prefix=filename_prefix,
                        filename_extension=filename_extension,
                    )
                ))
        pr.run()
        self.ctx["text_table_to_file"] = table_to_textfile
        if dumped:
            resource = util.get_or_create_resource(
                self,
                self.ctx.get("text_tables_resource_id"),
                description="description of diff between userdata tables %s and %s" % (self.ctx.get('state_a_resource_id'), self.ctx.get('state_b_resource_id')),
                resource_path=diff_folder,
                resource_type=resource_types.USERDATA_TABLES_DIFF_DESCR,
                arch='any',
            )
            self.mark_resource_ready(resource.id)
            self.ctx["text_tables_resource_id"] = resource.id
            self.ctx["text_tables_resource_url"] = resource.proxy_url

    @property
    def footer(self):
        if self.status != self.Status.SUCCESS:
            return

        if not self.ctx.get("has_diff"):
            return "<h3 style='color:green'>No diff</h3>"

        def get_text_link(table):
            if table in self.ctx.get("text_table_to_file", {}):
                return '{}/{}'.format(
                    self.ctx["text_tables_resource_url"],
                    self.ctx["text_table_to_file"][table]
                )
            return None

        def get_bin_link(table):
            if table in self.ctx.get("table_to_file", {}):
                return '{}/{}'.format(
                    self.ctx["tables_resource_url"],
                    self.ctx["table_to_file"][table]
                )
            return None

        def make_diff_entry(name, text_link=None, bin_link=None, is_new=False, is_old=False):
            assert not (is_new and is_old), "Diff entry could not be new and old in the same time"
            if is_new:
                html = "+ "
                color = "#e0ffe0"
            elif is_old:
                html = "- "
                color = "#ffe0e0"
            else:
                html = "* "
                color = "lemonchiffon"

            return "<span style='background-color: {}'>{}</span>".format(
                color,
                html + ('<a href="{}">[text]</a>'.format(text_link) if text_link else "") +
                ('<a href="{}">[bin]</a>'.format(bin_link) if bin_link else "") + " " + name
            )

        resources = []
        out = []
        htmls = []  # (name, html)
        diff_text = self.ctx.get("diff_text")
        if diff_text:
            text_link_base = self.ctx.get("diff_text_resource_url")
            if text_link_base:
                resources.append("<a href='{}'>files</a>".format(self.ctx.get("diff_text_resource_url", "")))

            htmls = []
            for file_name in sorted(diff_text["only_in_a"]):
                htmls.append(make_diff_entry(file_name, is_old=True))
            for file_name in sorted(diff_text["only_in_b"]):
                htmls.append(make_diff_entry(file_name, is_new=True))

            htmls_content_diff = []
            for file_name, content in diff_text["content"].iteritems():
                text_link = None
                if text_link_base and content.get("diff"):
                    text_link = "{}/{}".format(text_link_base, content["diff"])
                htmls_content_diff.append((file_name, make_diff_entry(file_name, text_link=text_link)))
            htmls_content_diff.sort(key=lambda h: h[0])  # sort by name
            htmls.extend(htmls_content_diff)

            out.append({
                "helperName": "",
                "content": {
                    "<h3 style='color:red'>Files diff</h3>": [
                        {"": h[1]}
                        for h in htmls
                    ],
                }
            })

        diff_tables = self.ctx.get('diff_tables', None)
        if diff_tables:
            # XXX: compatibility
            if not isinstance(diff_tables, list):
                diff_tables = json.loads(diff_tables)

            if "text_tables_resource_url" in self.ctx:
                resources.append("<a href='{}'>tables (text)</a>".format(self.ctx["text_tables_resource_url"]))
            if "tables_resource_url" in self.ctx:
                resources.append("<a href='{}'>tables (binary)</a>".format(self.ctx["tables_resource_url"]))

            htmls = []  # (name, html)
            old_prefix = self.get_prefix_for("diff/a_only")
            new_prefix = self.get_prefix_for("diff/b_only")
            changed_prefix = self.get_prefix_for("diff")
            for table in sorted(diff_tables):
                name = table
                for prefix in [old_prefix, new_prefix, changed_prefix]:
                    if name.startswith(prefix):
                        name = name[len(prefix):]
                        break
                htmls.append((name, make_diff_entry(name, text_link=get_text_link(table), bin_link=get_bin_link(table), is_new=table.startswith(new_prefix), is_old=table.startswith(old_prefix))))

            htmls.sort(key=lambda h: h[0])  # sort by name

            out.append({
                "helperName": "",
                "content": {
                    "<h3 style='color:red'>Tables diff</h3>": [
                        {"": h[1]}
                        for h in htmls
                    ],
                }
            })
        out.append({
            "helperName": "",
            "content": {
                "<h3 style='color:red'>Diff resources</h3>": [
                    {"": r}
                    for r in resources
                ],
            }
        })

        return out


__Task__ = CompareMRTables
