# -*- coding: utf-8 -*-

import os
import hashlib

import logging

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.projects import resource_types


class MatrixnetFirstTestParameter(ResourceSelector):
    name = 'matrixnet_first_test_resource_id'
    description = 'matrixnet first test results'
    resource_type = 'MATRIXNET_ALLTESTS_RESULTS'
    required = True


class MatrixnetSecondTestParameter(ResourceSelector):
    name = 'matrixnet_second_test_resource_id'
    description = 'matrixnet second test results'
    resource_type = 'MATRIXNET_ALLTESTS_RESULTS'
    required = True


def md5_for_file(f, block_size=2**20):
    md5 = hashlib.md5()
    while True:
        data = f.read(block_size)
        if not data:
            break
        md5.update(data)
    return str(md5.hexdigest())


def _get_files_md5(path):
    predictionFiles = os.listdir(path)
    result = dict()
    for f in predictionFiles:
        result[f] = md5_for_file(open(os.path.join(path, f), 'r'))
    return result


class CompareMatrixnets(SandboxTask):
    """
    **Описание**
    Сравнение моделей матрикснета между двумя ревизиями (на самом деле сравниваются предсказания на нескольких тестовых пулах, но это эквивалентно сравнению моделей)
    Результаты сравнения записываются в контекст

    **Необходимые ресурсы**

    * **matrixnet first test results** - архив с файлами предсказаний для различных тестов в ревизии N. ресурс типа MATRIXNET_ALLTESTS_RESULTS
    * **matrixnet second test results** - архив с файлами предсказаний для различных тестов в ревизии M. ресурс типа MATRIXNET_ALLTESTS_RESULTS
    """
    type = 'COMPARE_MATRIXNETS'

    input_parameters = (
        MatrixnetFirstTestParameter,
        MatrixnetSecondTestParameter,
    )

    def _write_md5_list_file(self, first_md5_sums, second_md5_sums, fileName):
        with open(fileName, 'w') as dst:
            for entry in first_md5_sums:
                if entry in second_md5_sums:
                    if first_md5_sums[entry] != second_md5_sums[entry]:
                        dst.write('diff found for test %s\n' % entry)
                        dst.write('\tfirst test md5\t%s\n' % first_md5_sums[entry])
                        dst.write('\tsecond test md5\t%s\n' % second_md5_sums[entry])
                        self.ctx['mx_diff'] = True

    def on_execute(self):
        if channel.sandbox.get_resource(self.ctx['matrixnet_first_test_resource_id']).file_md5 == channel.sandbox.get_resource(self.ctx['matrixnet_second_test_resource_id']).file_md5:
            logging.info('archives md5 match. implied ok')
            self.ctx['mx_diff'] = False
            return
        os.mkdir('alltests_first')
        os.mkdir('alltests_second')
        run_process('tar -C alltests_first -zxf %s --strip-components=1' % self.sync_resource(self.ctx['matrixnet_first_test_resource_id']), log_prefix='extract_first')
        run_process('tar -C alltests_second -zxf %s --strip-components=1' % self.sync_resource(self.ctx['matrixnet_second_test_resource_id']), log_prefix='extract_second')
        first_md5_sums = _get_files_md5('alltests_first')
        second_md5_sums = _get_files_md5('alltests_second')
        tests_diff_resource = self.create_resource('tests.diff', 'tests.diff', resource_types.MATRIXNET_ALLTESTS_DIFF)
        self.ctx['matrixnet_tests_diff_resource_id'] = tests_diff_resource.id
        self._write_md5_list_file(first_md5_sums, second_md5_sums, tests_diff_resource.path)
        if 'mx_diff' not in self.ctx:
            self.ctx['mx_diff'] = False


__Task__ = CompareMatrixnets
