# -*- coding: utf-8 -*-

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types

from .sub_requests import Requests
from .write_diff import html_diff


class Params:
    class Eventlog1(ResourceSelector):
        name = 'eventlog1_resource_id'
        description = 'Eventlog #1'
        resource_type = resource_types.EVENTLOG_DUMP
        required = True

    class Eventlog2(ResourceSelector):
        name = 'eventlog2_resource_id'
        description = 'Eventlog #2'
        resource_type = resource_types.EVENTLOG_DUMP
        required = True

    class EvlogdumpExecutable(ResourceSelector):
        name = 'evlogdump_executable_resource_id'
        description = 'evlogdump executable'
        resource_type = resource_types.EVLOGDUMP_EXECUTABLE
        required = True

    class RequestsPerFile(SandboxIntegerParameter):
        name = 'requests_per_file'
        description = 'requests number per 1 output file (*N subsources)'
        group = 'Diff options'
        default_value = 10

        @staticmethod
        def get_from(ctx):
            param = Params.RequestsPerFile
            v = ctx.get(param.name, param.default_value)
            if v == 0:
                v = param.default_value
            return v

    input_parameters = (
        Eventlog1,
        Eventlog2,
        EvlogdumpExecutable,
        RequestsPerFile,
    )


class CompareSubSourceRequests(SandboxTask):
    """
        Compare subsource requests cgi parameters (with same reqid)
    """

    type = 'COMPARE_METASEARCH_SUBSOURCE_REQUESTS'
    input_parameters = Params.input_parameters

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self
        self.ctx['out_resource_html_id'] = self.create_resource(
            "{}, html diff".format(self.descr), 'diff',
            resource_types.METASEARCH_SUBSOURCE_REQUESTS_COMPARE_RESULT,
            attributes={
                "ttl": 90,
            }
        ).id

    parsed_eventlog1 = None
    parsed_eventlog2 = None

    def on_execute(self):
        path_to_diffs = channel.sandbox.get_resource(self.ctx['out_resource_html_id']).path
        make_folder(path_to_diffs)

        evlogdump_binary = self.sync_resource(self.ctx[Params.EvlogdumpExecutable.name])

        # parse binary eventlogs to text format
        evlog1 = self.sync_resource(self.ctx[Params.Eventlog1.name])
        self.parsed_eventlog1 = self.abs_path('eventlog1.txt')
        evlog2 = self.sync_resource(self.ctx[Params.Eventlog2.name])
        self.parsed_eventlog2 = self.abs_path('eventlog2.txt')
        with open(evlog1) as in_file1, open(self.parsed_eventlog1, "w") as out_file1:
            process.run_process([evlogdump_binary], stdin=in_file1, stdout=out_file1)
        with open(evlog2) as in_file2, open(self.parsed_eventlog2, "w") as out_file2:
            process.run_process([evlogdump_binary], stdin=in_file2, stdout=out_file2)

        # parse text eventlogs to Requests
        reqs1 = Requests(self.parsed_eventlog1)
        reqs2 = Requests(self.parsed_eventlog2)

        data = []  # <- put here requests with same reqid
        for reqid, req1 in reqs1.idx.iteritems():
            req2 = reqs2.idx.get(reqid, None)
            if req2:
                data.append((reqid, req1.node, req2.node))

        # write diff as html files
        res = html_diff(path_to_diffs, data, Params.RequestsPerFile.get_from(self.ctx))

        self.ctx["diff_count"] = res.diff_count
        self.ctx["compare_count"] = res.compare_count
        self.mark_resource_ready(self.ctx['out_resource_html_id'])


__Task__ = CompareSubSourceRequests
