# -*- coding: utf-8 -*-
from __future__ import division

import os
import json
from urllib import unquote
from itertools import izip

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common import errors
from sandbox.common.types.resource import State
from sandbox.projects.common.wizard import utils as wizard_utils
from sandbox.projects.websearch.begemot.tasks.BegemotCreateResponsesDiff import jsondiff


RULES = ['qtree', 'user_request', 'relevgeo', 'Cho', 'dmoz_th', 'diversity_categ',
         'qrr', 'th3973', 'cm2', 'avgl', 'hodi', 'ilp', 'tc', 'qr2v', 'ispicture',
         'po', 'qr2u', 'cm', 'qr2r', 'Cdi2', 'ydngi', 'qradm', 'norm', 'fe',
         'relev_locale', 'wminone', 'tv', 'qc', 'istext', 'qrv', 'di2', 'mu',
         'navmx', 'qrcty', 'isorg', 'wmaxone', 'Cavgl', 'is_nav', 'rnq', 'isgeo',
         'th3561', 'di', 'gc', 'Cpo', 'Chodi', 'cntry_r', 'ho', 'qlang', 'pr',
         'prel', 'ismusic', 'isurl', 'hpq', 'syq', 'Cdi', 'ishum', 'qru', 'vq',
         'fqwg', 'qrnb']


def wizard_cgi_to_json(wiz_resp, full_response):
    json = {}
    for cgi_param in wiz_resp.split('&'):
        key, value = unquote(cgi_param).split('=', 1)
        if key in ['relev', 'rearr']:
            json[key] = {}
            for relev in value.split(';'):
                relev_key, relev_value = relev.split('=', 1)
                if relev_key in RULES or full_response:
                    json[key][relev_key] = relev_value.decode('utf-8')
        elif key in RULES or full_response:
            json[key] = value.decode('utf-8')
    return json


class CompareProdAndRsyaResps(sdk2.Task):
    """
        Compares wizard cgi answers
        Is used to check that diff between service wizard and rsya wizard is less than 0.04%
        and diff between debug and release wizard binaries is less than 0.3%
    """
    unpack_rich_tree_tool = None

    class Requirements(sdk2.Requirements):
        client_tags = wizard_utils.ALL_SANDBOX_HOSTS_TAGS

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 10
        kill_timeout = 3600
        responses_rsya_conf = sdk2.parameters.Resource(
            'Responses from wizard with rsya conf',
            resource_type=resource_types.WIZARD_RESPONSES_RESULT,
            required=True
        )
        responses_prod_conf = sdk2.parameters.Resource(
            'Responses from wizard with prod conf',
            resource_type=resource_types.WIZARD_RESPONSES_RESULT,
            required=True
        )
        compare_full_response = sdk2.parameters.Bool(
            'Compare full response', default=False,
            description='If false, only fields which affect rsya wizard will be compared',
        )
        acceptable_diff_percent = sdk2.parameters.Float('Acceptable diff percent', default=0.04)

    def on_save(self):
        wizard_utils.setup_hosts(self)

    def unpack_rich_tree(self, base64qtree):
        try:
            r, w = os.pipe()
            os.write(w, base64qtree)
            os.close(w)
            output = sp.check_output([self.unpack_rich_tree_tool], stdin=r)
        except sp.CalledProcessError:
            self.set_info('unpackrichtree failed\nInput:\n%s' % base64qtree)
            raise
        try:
            return json.loads(output.decode('utf-8'))
        except (ValueError, UnicodeDecodeError) as e:
            if isinstance(e, UnicodeDecodeError):
                self.set_info('cannot decode to utf-8 unpackrichtree output: %s' % output)
            if isinstance(e, ValueError):
                self.set_info('cannot parse unpackrichtree output to json: %s' % output)
            self.set_info('qtree: %s' % base64qtree)
            raise

    def on_execute(self):
        r = resource_types.UNPACKRICHTREE.find(state=State.READY).order(-sdk2.Resource.id).first()
        if r:
            self.unpack_rich_tree_tool = str(sdk2.ResourceData(r).path)
        else:
            self.set_info('resource with unpackrichtree tool not found, base64-encoded qtree will be compared')

        response_file_prod = str(sdk2.ResourceData(self.Parameters.responses_rsya_conf).path)
        response_file_rsya = str(sdk2.ResourceData(self.Parameters.responses_prod_conf).path)

        matches = 0
        total = 0
        with open(response_file_prod) as f1, open(response_file_rsya) as f2, open('diff.txt', 'w') as fp3:
            for l1, l2 in izip(f1, f2):
                ans1 = wizard_cgi_to_json(l1.strip(), self.Parameters.compare_full_response)
                ans2 = wizard_cgi_to_json(l2.strip(), self.Parameters.compare_full_response)
                diff = jsondiff.diff(ans1, ans2)
                if diff:
                    if total - matches < 100:
                        if diff['qtree']:
                            diff['qtree'] = jsondiff.diff(
                                self.unpack_rich_tree(ans1['qtree']), self.unpack_rich_tree(ans2['qtree'])
                            )
                        fp3.write(''.join(jsondiff.render_text([('Old', 'New', diff)])).encode('utf-8'))
                else:
                    matches += 1
                total += 1
        resource_types.TASK_LOGS(self, 'Responses diff', 'diff.txt')

        diff_rate = 1 - matches / total if total else 0
        diff_rate *= 100
        if diff_rate > self.Parameters.acceptable_diff_percent:
            raise errors.TaskFailure(
                'Diff between RSYA and prod config is greater than '
                '{}% ({:.2f}%)'.format(self.Parameters.acceptable_diff_percent, diff_rate)
            )
        self.set_info("Diff = {:.2f}%".format(diff_rate))
