# -*- coding: utf-8 -*-
import os
import imp
import sys
import shutil

from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils


class InputPool1(sp.ResourceSelector):
    name = 'final_pool_1'
    required = True
    description = 'pool 1'
    resource_type = resource_types.IDX_OPS_EST_FEATURES_PROTOBUF_FINAL_OUTPUT
    group = 'Task resources'


class InputPool2(sp.ResourceSelector):
    name = 'final_pool_2'
    required = True
    description = 'pool 2'
    resource_type = resource_types.IDX_OPS_EST_FEATURES_PROTOBUF_FINAL_OUTPUT
    group = 'Task resources'


class Pool1Names(sp.ResourceSelector):
    name = 'names_1'
    required = True
    description = 'feature names for pool 1'
    resource_type = resource_types.FACTOR_NAMES_TXT
    group = 'Task resources'


class Pool2Names(sp.ResourceSelector):
    name = 'names_2'
    required = True
    description = 'feature names for pool 2'
    resource_type = resource_types.FACTOR_NAMES_TXT
    group = 'Task resources'


class ComparerExecutable(sp.LastReleasedResource):
    name = 'comparer_executable'
    required = True
    description = 'Comparer executable'
    resource_type = resource_types.POOL_COMPARER_EXECUTABLE
    group = 'Custom params'


class AddDetailedJsons(sp.SandboxBoolParameter):
    name = 'add_detailed_jsons'
    required = False
    description = 'Save detailed info'
    group = 'Custom params'
    default_value = False


class CompareProtoPools(SandboxTask):
    """
    Сравнивает два protobuf-final выхлопa ``idx_ops`` в режиме ``estfeatures2f``
    на количество обработанных текстовой машиной хитов.
    """
    type = 'COMPARE_PROTO_POOLS'

    input_parameters = (
        InputPool1,
        InputPool2,
        Pool1Names,
        Pool2Names,
        ComparerExecutable,
        AddDetailedJsons,
    )
    cores = 1
    execution_space = 15 * 1024  # 15 Gb

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        resource = self._create_resource(self.descr, 'compare_result', resource_types.TM_LIMITS_MONITOR_COMPARE_RESULT)
        self.ctx['out_resource_id'] = resource.id

    def load_converter(self, binary):
        # r = subprocess.check_output([binary, "--svnrevision"], shell=False)

        # for t in r.split("\n"):
        #     if t.find("URL") > 0:
        #         path = t.split(" ")[-1]
        #     # if t.find("Rev:") > 0:
        #         # revision = t.split(" ")[-1]

        Arcadia.export(
            Arcadia.trunk_url('/search/tools/idx_ops/comparer/to_html_converter/pool_comparer_result_converter.py'),
            # path + "/search/tools/idx_ops/comparer/to_html_converter/pool_comparer_result_converter.py",
            self.abs_path('pool_comparer_result_converter.py'),
            # revision = revision,
        )

        sys.path.append(self.abs_path("./"))
        fp, pathname, description = imp.find_module("pool_comparer_result_converter")
        res = imp.load_module("pool_comparer_result_converter", fp, pathname, description)
        sys.path.pop()
        return res

    def on_execute(self):
        pool1 = self._read_resource(self.ctx[InputPool1.name])
        pool2 = self._read_resource(self.ctx[InputPool2.name])
        names1 = self._read_resource(self.ctx[Pool1Names.name])
        names2 = self._read_resource(self.ctx[Pool2Names.name])
        comparer = self._read_resource(self.ctx[ComparerExecutable.name])

        resource = self._read_resource(self.ctx['out_resource_id'], sync=False)
        make_folder(resource.abs_path())

        comparer_results_path = os.path.join(resource.abs_path(), "detailed_jsons")
        detailed_stats_path = os.path.join(resource.abs_path(), "detailed_stats")
        make_folder(comparer_results_path)
        make_folder(detailed_stats_path)

        comparer_results = {
            "bl": os.path.join(comparer_results_path, "bl_stats"),
            "so": os.path.join(comparer_results_path, "so_stats"),
            "agg": os.path.join(comparer_results_path, "agg_diff"),
            "line": os.path.join(comparer_results_path, "line_diff"),
            "ds": os.path.join(comparer_results_path, "diff_stats"),
        }

        converter_results = {
            "bl": os.path.join(detailed_stats_path, "baseline_stats.html"),
            "so": os.path.join(detailed_stats_path, "test_stats.html"),
            "agg": os.path.join(detailed_stats_path, "aggregations_diff.html"),
            "line": os.path.join(resource.abs_path(), "per_line_diff_features.html"),
            "line_full": os.path.join(resource.abs_path(), "per_line_diff_hits.html"),
            "ds": os.path.join(resource.abs_path(), "diff_stats.html"),
        }

        process.run_process(
            [
                comparer.abs_path(),
                "-i", pool1.abs_path(),
                "-t", pool2.abs_path(),
                "--dst-bl", comparer_results["bl"],
                "--dst-so", comparer_results["so"],
                "--dst-diff-agg", comparer_results["agg"],
                "--dst-diff-line", comparer_results["line"],
                "--dst-diff-stats", comparer_results["ds"],
            ],
            log_prefix='run comparer',
            check=True,
        )

        with open(names1.abs_path()) as f:
            bl_names = [x.strip() for x in f.readlines()]

        with open(names2.abs_path()) as f:
            so_names = [x.strip() for x in f.readlines()]

        pc = self.load_converter(comparer.abs_path())

        ld = pc.TPrintLinedDiff(comparer_results["line"], None, True, bl_names, so_names)
        ld.print_lined_diff(converter_results["line"])

        ld2 = pc.TPrintLinedDiff(comparer_results["line"], None, False, bl_names, so_names)
        ld2.print_lined_diff(converter_results["line_full"])

        ad = pc.TPrintAggDiff(comparer_results["agg"], None, bl_names, so_names)
        ad.print_aggregated_diff(converter_results["agg"])

        differed_features = ld.get_differed_f_keys()
        differed_features.update(ad.get_differed_f_keys())

        changed_features_list = [
            pc.NamesResolve(x, {"info": differed_features[x]}, bl_names, so_names, None)["full_name"]
            for x in differed_features
        ]

        if ld2.difference_in_hits:
            changed_features_list.append("TMHits")
        else:
            os.remove(converter_results["line_full"])

        if len(differed_features) > 40:
            diff_features_list_resource = self._create_resource(self.descr + ": changed factors list", 'changed_pool_factors_list', resource_types.OTHER_RESOURCE)
            with open(diff_features_list_resource.abs_path(), "w") as f:
                print >> f, changed_features_list
            self.ctx['changed_factors'] = ["number of changed factors in pool is %d, see details in resource %d" % (len(differed_features), diff_features_list_resource.id)]
        else:
            self.ctx['changed_factors'] = changed_features_list

        bl = pc.TPrintStats(comparer_results["bl"], False, None, bl_names, so_names, differed_features)
        bl.print_stats(converter_results["bl"], "BaseLine Stats")

        sl = pc.TPrintStats(comparer_results["so"], False, None, bl_names, so_names, differed_features)
        sl.print_stats(converter_results["so"], "Testing one Stats")

        bl = pc.TPrintStats(comparer_results["ds"], True, None, bl_names, so_names, differed_features)
        bl.print_stats(converter_results["ds"], "Stats of difference")

        if not utils.get_or_default(self.ctx, AddDetailedJsons):
            shutil.rmtree(comparer_results_path)

        if not ad.has_diff and not ld.has_diff:
            fu.write_file(
                os.path.join(resource.abs_path(), 'noDifferences.txt'),
                'There are no differences between pools or '
                'pools are not intersected and all aggregation stats differ by less than 0.5%'
            )

        self.ctx['changed_aggregated_stats'] = ad.has_diff
        self.ctx['per_line_diff'] = ld.has_diff

        size1 = os.path.getsize(pool1.abs_path())
        size2 = os.path.getsize(pool2.abs_path())

        if size1 == 0 or size2 == 0:
            eh.check_failed("Incorrect file size")

        self.ctx['output_diff'] = 1.0 * size1 / size2

        resource.mark_ready()

    def get_short_task_result(self):
        if self.is_completed():
            res = ""
            if self.ctx['per_line_diff']:
                res = "diff"

            if self.ctx['changed_aggregated_stats']:
                res = res + ("" if res == "" else " ") + "agg_diff"

            if res == "":
                res = "no diff"
            return res

        return None


__Task__ = CompareProtoPools
