from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import process
from sandbox.projects.common import link_builder as lb
from sandbox.projects.websearch.upper import resources as upper_res


_HAS_DIFF_KEY = 'has_diff'


class OldUnistatSignals(sp.ResourceSelector):
    name = 'old_unistat_signals_resource_id'
    description = 'Old unistat signals'
    resource_type = [
        resource_types.BASESEARCH_UNISTAT_SIGNALS,
        upper_res.UpperUnistatSignals,
    ]
    required = True


class NewUnistatSignals(sp.ResourceSelector):
    name = 'new_unistat_signals_resource_id'
    description = 'New unistat signals'
    resource_type = [
        resource_types.BASESEARCH_UNISTAT_SIGNALS,
        upper_res.UpperUnistatSignals,
    ]
    required = True


class CompareUnistat(SandboxTask):
    """
        Compare two sets of unistat signals
    """
    type = 'COMPARE_UNISTAT'

    input_parameters = (OldUnistatSignals, NewUnistatSignals)
    cores = 1
    execution_space = 50  # 50M is enough according to sandbox info

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        self.ctx["kill_timeout"] = 600  # 10min is enough
        resource = self._create_resource(self.descr, 'dependencies.diff', resource_types.UNIFIED_GNU_DIFF_FILE)
        self.ctx['out_resource_id'] = resource.id

    def on_execute(self):
        old_output = self.sync_resource(self.ctx[OldUnistatSignals.name])
        new_output = self.sync_resource(self.ctx[NewUnistatSignals.name])

        diff_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])
        with open(diff_resource.path, 'w') as diff_stdout:
            diff_process = process.run_process(
                ['diff', '-u', old_output, new_output],
                stdout=diff_stdout,
                outputs_to_one_file=False,
                log_prefix="diff",
                check=False,
            )
            self.ctx[_HAS_DIFF_KEY] = diff_process.returncode != 0
        if self.ctx.get(_HAS_DIFF_KEY):
            link_to_diff = lb.HREF_TO_ITEM.format(link=diff_resource.proxy_url, name="Link to diff")
            self.set_info(link_to_diff, do_escape=False)
        else:
            self.set_info("No diff detected")

    def get_short_task_result(self):
        if not self.is_completed():
            return None
        return "diff" if self.ctx.get(_HAS_DIFF_KEY, False) else "no diff"


__Task__ = CompareUnistat
