import json
import logging
import re
import traceback

import sandbox.common.types.client as ctc
import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.projects.common.YMakeBase import YMakeCache

import child
import compare


class CompareYmakeDump(sdk2.Task):
    """ Run ymake 2 times and compare results """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_PRECISE
        cores = 1
        ram = 20 * 1024
        disk_space = 32000
        tasks_resource = None

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 4200
        max_restarts = 1

        # custom parameters

        with sdk2.parameters.Group('Common parameters') as common_options:
            gen_conf_opts = sdk2.parameters.List('Options for "ya dump conf" (for 1st and 2nd run)', value_type=sdk2.parameters.String, default=[])
            targets = sdk2.parameters.List('Build targets (for 1st and 2nd run)', value_type=sdk2.parameters.String, default=[])
            hash_salt = sdk2.parameters.String('Task type (for cache hash calculation)', required=True, default='Manual run')
            use_arcadia_api = sdk2.parameters.Bool('Use arcadia-api fuse', default=True)

            with sdk2.parameters.RadioGroup('Default ymake') as default_ymake_type:
                default_ymake_type.values['ya_tool'] = default_ymake_type.Value(value='ya tool ymake', default=True)
                default_ymake_type.values['last_ymake'] = default_ymake_type.Value(value='last YMAKE_BINARY')

        with sdk2.parameters.Group('First run parameters') as first_run:
            arcadia_url_old = sdk2.parameters.ArcadiaUrl('Arcadia to run ymake on (for 1st run)', required=True)
            ymake_bin_old = sdk2.parameters.Resource('YMake binary (for 1st run)', resource_type=resource_types.YMAKE_BINARY, required=False)

        with sdk2.parameters.Group('Second run parameters') as second_run:
            arcadia_url_new = sdk2.parameters.ArcadiaUrl('Arcadia to run ymake on (for 2nd run)', required=True)
            ymake_bin_new = sdk2.parameters.Resource('YMake binary (for 2nd run)', resource_type=resource_types.YMAKE_BINARY, required=False)

            use_cache = sdk2.parameters.Bool('Use cache (for 2nd run)')
            with use_cache.value[True]:
                use_fs_cache_only = sdk2.parameters.Bool('Use only filesystem-cache part of internal cache')
            use_json_cache = sdk2.parameters.Bool('Use JSON cache (for 2nd run)')
            use_patch_mode = sdk2.parameters.Bool('Use ymake patch mode (for 2nd run)')
            with use_patch_mode.value[True]:
                use_arc_changelist = sdk2.parameters.Bool('Use arc for patch generating')
                fail_if_patch_unavailable = sdk2.parameters.Bool('Fail if patch is unavailable')
            cache_resource = sdk2.parameters.Resource('Result of a previous run (for 2nd run)', resource_type=YMakeCache, required=False)

        with sdk2.parameters.Group('Diff parameters') as diff_options:
            generate_diff = sdk2.parameters.Bool('Generate diff')
            context_size = sdk2.parameters.Integer('Lines in unified context for diff', required=True, default=10)
            generate_html_diff = sdk2.parameters.Bool('Generate HTML diff')
            compare_darts = sdk2.parameters.Bool('Dump and compare darts')
            fail_if_diff = sdk2.parameters.Bool('Fail if graphs or darts differ')
            fail_if_error = sdk2.parameters.Bool('Fail instead of exception')

    class Context(sdk2.Task.Context):
        fresh_run_task_id = None
        dry_run_task_id = None
        discovered_cache_id = None
        has_diff = False
        has_json_diff = False
        has_json_diff_without_uids = False
        has_internal_diff = False
        has_dart_diff = False
        report = None
        metrics = None

    def dump_metrics(self):
        fresh_run_metrics = json.loads(sdk2.Task[self.Context.fresh_run_task_id].Context.metrics)
        dry_run_metrics = json.loads(sdk2.Task[self.Context.dry_run_task_id].Context.metrics)
        metrics = {}
        for fresh_item, dry_item in zip(fresh_run_metrics, dry_run_metrics):
            name = fresh_item[0].lower().replace(' ', '_')
            metrics['fresh_' + name] = fresh_item[1]
            metrics['dry_' + name] = dry_item[1]
        self.Context.metrics = json.dumps(metrics)

    def check_parameters(self):
        if self.Parameters.generate_html_diff and not self.Parameters.generate_diff:
            raise common.errors.TaskFailure('"Generate HTML diff" requires "Generate diff"')

    @sdk2.report(title="Comparison report")
    def report(self):
        return re.sub('\n', '<br>', self.Context.report) if self.Context.report else "..."

    def on_create(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={'release': ctt.ReleaseStatus.STABLE,
                   'target': 'sandbox/projects/CompareYmakeDump'}
        ).first()

    def on_execute(self):
        self.check_parameters()

        if child.need_create_children(self):
            children = child.create_all(self)
            raise sdk2.WaitTask(children, [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK], wait_all=True)

        logging.info('All subtasks have finished')

        try:
            self.dump_metrics()
            compare.compare_all(self)
            compare.save_results(self)

        except Exception:
            if self.Parameters.fail_if_error:
                logging.error(traceback.format_exc())
                raise common.errors.TaskFailure('Exception during results processing')
            else:
                raise

        if self.Parameters.fail_if_diff and self.Context.has_diff:
            raise common.errors.TaskFailure('Graphs differ')
