import enum
import filecmp
import json
import logging
import os
import re
import shutil
import StringIO

from sandbox import common
from sandbox import sdk2
from sandbox.projects.common import differ as diff


class YMakeDumpsDiff(sdk2.Resource):
    """ Contains diffs from 2 ymake runs """

    releasable = False
    any_arch = True


def remove_ids(input_path, output_path):
    with open(input_path, 'r') as in_file:
        data = in_file.readlines()
    with open(output_path, 'w') as out_file:
        data = filter(lambda x: 'SOURCE-FOLDER-PATH' not in x, data)
        data = [re.sub('Id: \d*', 'Id: x', x) for x in data]
        out_file.writelines(data)
    return output_path


def clean_dart(input_path, output_path):
    data = open(input_path, 'r').readlines()
    data = filter(lambda x: 'SOURCE-FOLDER-PATH' not in x, data)
    open(output_path, 'w').writelines(data)
    return output_path


def format_json(input_path, output_path):
    with open(input_path, 'r') as in_file:
        data = in_file.read()
    with open(output_path, 'w') as out_file:
        obj = json.loads(data)
        obj['graph'].sort(key=lambda node: node['outputs'][0])
        obj['result'].sort()
        out_file.write(json.dumps(obj, sort_keys=True, indent=1))
    return output_path


def remove_uids(dump_file, result_dir, postfix):
    pretty_dump = os.path.join(result_dir, os.path.basename(dump_file) + postfix)
    with open(dump_file, 'r') as in_file:
        data = in_file.read()
    with open(pretty_dump, 'w') as out_file:
        data = re.sub('"' + '[\w-]'*22 + '"', '"uid"', data)
        out_file.write(data)
    return pretty_dump


class DumpType(enum.Enum):
    Internal, Json, JavaDart, TestDart, MakefilesDart = range(5)


def get_results_dir(task, child_task_id):
    task = sdk2.Task[child_task_id]
    data = sdk2.ResourceData(sdk2.Resource[task.Context.dumps_resource_id])
    return str(data.path)


def compare_dumps(task, file1, file2, dump_type):
    prepare = None
    if dump_type == DumpType.Internal:
        prepare = remove_ids
    elif dump_type == DumpType.Json:
        prepare = format_json
    else:
        prepare = clean_dart
    file1 = prepare(file1, str(task.path('1.' + dump_type.name + '.formatted')))
    file2 = prepare(file2, str(task.path('2.' + dump_type.name + '.formatted')))

    equal = filecmp.cmp(file1, file2)
    task.Context.has_diff |= not equal
    task.Context.has_json_diff |= dump_type == DumpType.Json and not equal
    task.Context.has_internal_diff |= dump_type == DumpType.Internal and not equal
    task.Context.has_dart_diff |= dump_type in (DumpType.JavaDart, DumpType.MakefilesDart, DumpType.TestDart) and not equal
    logging.info(file1 + ' ' + file2 + (' are equal' if equal else ' differ'))

    if equal:
        map(os.remove, (file1, file2))
        return

    if not task.Parameters.generate_diff:
        return

    if dump_type == DumpType.Json:
        file1 = remove_uids(file1, str(task.path()), '')
        file2 = remove_uids(file2, str(task.path()), '')

    diff_name = dump_type.name + '.diff'

    p = sdk2.helpers.subprocess.Popen(['diff', '-U', str(task.Parameters.context_size), file1, file2],
                                      stdout=sdk2.helpers.subprocess.PIPE,
                                      stderr=sdk2.helpers.subprocess.PIPE,
                                      )
    out, err = p.communicate()

    if p.returncode == 0:
        map(os.remove, (file1, file2))
        return
    elif p.returncode == 1:
        if dump_type == DumpType.Json:
            task.Context.has_json_diff_without_uids = True

        with open(str(task.path(diff_name)), 'w') as f:
            f.write(out)

        if not task.Parameters.generate_html_diff:
            return

        if len(out) > 1 * 1000 * 1000:
            logging.info('Diff is too big to be pretty-printed')
            return

        if dump_type == DumpType.Json:
            data1 = open(file1, 'r').read()
            data2 = open(file2, 'r').read()

            printer = diff.printers.PrinterToHtml(str(task.path()), write_compact_diff=True)
            differ = diff.json_differ.JsonDiffer(printer)
            differ.compare_single_pair(data1, data2)
            printer.finalize()

            shutil.move(str(task.path('r_000000-000000.html')), str(task.path(diff_name + '.html')))
        else:
            import diff2html
            diff_input = StringIO.StringIO(out)
            with open(str(task.path(diff_name + '.html')), 'w') as f:
                diff2html.parse_input(diff_input, f, diff_name, diff_name + '.html', None, None)

    elif p.returncode == 2:
        raise common.errors.TaskFailure('Error in diff')


def save_results(task):
    resource = YMakeDumpsDiff(
        task=task,
        description='Diffs from CompareYmakeDump task #{}'.format(str(task.id)),
        path='diffs'
    )

    data = sdk2.ResourceData(resource)
    data.path.mkdir(0o755, parents=True, exist_ok=True)
    resource_dir = str(data.path)

    if task.Context.has_diff:
        files = sdk2.paths.list_dir(str(task.path()), files_only=True, abs_path=True)
        for path in files:
            path = str(path)
            _, ext = os.path.splitext(path)
            if ext in ('.diff', '.html', '.css', '.js', '.formatted'):
                shutil.move(path, os.path.join(resource_dir, os.path.basename(path)))

    report_path = os.path.join(resource_dir, 'report.txt')
    report = ''
    if not task.Context.has_diff:
        report += 'Results are identical\n'
    else:
        report += 'Internal graphs differ\n' if task.Context.has_internal_diff else 'Internal graphs are equal\n'
        report += 'JSON graphs differ\n' if task.Context.has_json_diff else 'JSON graphs are equal\n'
        report += 'JSON graphs differ (after remove_uids)\n' if task.Context.has_json_diff_without_uids else 'JSON graphs are equal (after remove_uids)\n'
    with open(report_path, 'w') as f:
        f.write(report)
    task.Context.report = report
    task.Context.diffs_resource_id = resource.id
    data.ready()


def compare_all(task):
    results_fresh = get_results_dir(task, task.Context.fresh_run_task_id)
    results_dry = get_results_dir(task, task.Context.dry_run_task_id)

    logging.info('Compare internal graphs')
    compare_dumps(task, os.path.join(results_fresh, 'ymake.out.log'), os.path.join(results_dry, 'ymake.out.log'), DumpType.Internal)

    logging.info('Compare JSON graphs')
    compare_dumps(task, os.path.join(results_fresh, 'graph.json'), os.path.join(results_dry, 'graph.json'), DumpType.Json)

    if task.Parameters.compare_darts:
        logging.info('Compare darts')
        for name, dump_type in (('java.dart', DumpType.JavaDart), ('test.dart', DumpType.TestDart), ('makefiles.dart', DumpType.MakefilesDart)):
            compare_dumps(task, os.path.join(results_fresh, name), os.path.join(results_dry, name), dump_type)
