import tarfile
import os
import logging

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.common.fs import untar_archive
from sandbox.projects.common.build.BuildForAllTask import DoNotRemoveResourcesParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.projects.common import apihelpers


class MSVCVersion(SandboxStringParameter):
    name = 'studio_version'
    description = 'Version of Visual Studio (2013, 2015 etc)'
    default_value = '2013'


class CreateCrossMSVCToolkit(SandboxTask):
    type = 'CREATE_CROSS_MSVC_TOOLKIT'

    input_parameters = [DoNotRemoveResourcesParameter, MSVCVersion]
    release_to = 'sandbox-releases@yandex-team.ru'

    execution_space = 4 * 1024

    def init_resources(self):
        wine_resource = apihelpers.get_last_resource_with_attribute(resource_types.SCRIPT_BUILD_RESULT, attribute_name='tool_name', attribute_value='wine')
        self.wine_path = channel.task.sync_resource(wine_resource)
        self.ver = self.ctx.get(MSVCVersion.name, 'UNDEFINED')
        msvc_ver = 'cross_{}'.format(self.ver)
        logging.info("Get resource for msvc with version: %s", msvc_ver)
        msvc_resource = apihelpers.get_last_resource_with_attribute(resource_types.VISUAL_CPP_COMPILER_TOOLKIT, attribute_name='msvc_compiler', attribute_value=msvc_ver)
        self.msvc_path = channel.task.sync_resource(msvc_resource)

    def on_execute(self):
        self.init_resources()
        # mkdir with result; untar wine and msvc. repack
        room_path = self.abs_path()
        msvc_untared_path = os.path.join(room_path, 'msvc')
        untar_archive(self.msvc_path, msvc_untared_path)
        wine_untared_path = os.path.join(room_path, 'wine')
        untar_archive(self.wine_path, wine_untared_path)

        native_dlls_list = os.path.join(msvc_untared_path, 'native_dll.list')
        if os.path.exists(native_dlls_list):
            logging.info('Rename stubs in wine from list: %s.', native_dlls_list)
            with open(native_dlls_list, 'r') as f:
                wine_stubs_path = os.path.join(wine_untared_path, 'lib64', 'wine')
                for line in f.readlines():
                    old_path = os.path.join(wine_stubs_path, line.strip() + '.so')
                    new_path = old_path + '.bak'
                    try:
                        os.rename(old_path, new_path)
                    except OSError:
                        logging.warning('No rename for %s.', old_path)
                        pass

        output_path = os.path.join(room_path, 'msvc_toolkit.tgz')
        with tarfile.open(output_path, "w:gz") as tar:
            tar.add(wine_untared_path, 'wine')
            tar.add(msvc_untared_path, 'msvc')

        self.create_resource(
            description='MSVC with Wine',
            resource_path=output_path,
            resource_type=resource_types.CROSS_MSVC_TOOLKIT,
            arch='any',
            attributes={'platform': 'win32', 'version': self.ver},
        )


__Task__ = CreateCrossMSVCToolkit
