import os
import time
import shutil
import logging
import datetime
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.sandboxapi as sdk_sandboxapi
from sandbox.projects.common.yabs.graphite import Graphite, five_min_metric, YABS_SERVERS

from sandbox.projects.common import apihelpers
from sandbox.projects import resource_types

from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common.apihelpers import get_last_released_resource

DATETIME_FORMAT = '%Y%m%d%H%M'


def get_latest_resource(resource_type, all_attrs=None):

    all_attrs = all_attrs or {}
    resources = channel.sandbox.list_resources(
        order_by="-id", limit=1, status="READY", resource_type=resource_type, all_attrs=all_attrs)

    if resources:
        return resources[0]
    logging.warning("Can't find latest resource: %s", resource_type)


def dump_previous_state_to_file(
    sandbox_task, filename,
    explicit_state_parameter, resource_type, attrs=None
):
    """
    :param sandbox_task:
    :param filename:
    :param explicit_state_parameter:
    :param resource_type:
    :param attrs:
    :return:
    """

    # in case state resource is is explicitly specified, use it
    state_resource_id = sandbox_task.ctx.get(explicit_state_parameter)

    if not state_resource_id:
        resource = get_latest_resource(
            resource_type=resource_type,
            all_attrs=attrs
        )
        if resource:
            state_resource_id = resource.id

    if not state_resource_id:
        raise ValueError

    shutil.copyfile(sandbox_task.sync_resource(state_resource_id), filename)


def save_state(sandbox_task, filename, resource_type, owner, last_log_time):

    ml_task_id = sandbox_task.ctx[sandbox_task.MLTaskID.name]

    sandbox_task.create_resource(
        "Nirvana online learning state for ml_task_id %s" % ml_task_id,
        filename,
        resource_type,
        owner=owner,
        attributes={
            "ml_task_id": ml_task_id,
            "log_time": last_log_time,
            "last_log_date": last_log_time
        }
    )


class OnlineLearningDumpTXTPrepare(sdk_task.SandboxTask):

    cores = 1

    class MLTaskID(sdk_parameters.SandboxStringParameter):
        name = "ml_task_id"
        description = "ml task, used to retrieve state"
        required = True

    class TaskParamsConf(sdk_parameters.ResourceSelector):
        name = "task_yaml"
        description = "yaml with graph parameters"
        required = True

    class TaskHoursBefore(sdk_parameters.SandboxIntegerParameter):
        name = 'hours_before'
        description = "hours for computing first feature map"
        required = True

    class YTTokenOwner(sdk_parameters.SandboxStringParameter):
        name = "yt_token_owner"
        description = "YT token owner"
        required = True
        default_value = "ML-ENGINE"

    class YTTokenName(sdk_parameters.SandboxStringParameter):
        name = "yt_token_name"
        description = "YT token name"
        required = True
        default_value = "robot_ml_engine_hahn_yt_token"

    class YQLTokenOwner(sdk_parameters.SandboxStringParameter):
        name = "yql_token_owner"
        description = "YQL token owner"
        required = True
        default_value = "ML-ENGINE"

    class YQLTokenName(sdk_parameters.SandboxStringParameter):
        name = "yql_token_name"
        description = "YQL token name"
        required = True
        default_value = "robot-ml-engine_ml_engine_app_token"

    class YTProxy(sdk_parameters.SandboxStringParameter):
        name = "yt_proxy"
        description = "YT proxy"
        required = True
        default_value = "hahn"

    class Stable(sdk_parameters.SandboxBoolParameter):
        name = "is_stable"
        description = "When true use stable binary from sandbox, when false use testing"
        default_value = True
        required = False

    class FixedBinaryID(sdk_parameters.ResourceSelector):
        name = "fixed_binary_id"
        description = "If you want to launch task with particular resource id"
        default_value = None
        required = False

    class MaxDelayHours(sdk_parameters.SandboxIntegerParameter):
        name = "max_delay_hours"
        description = "Max hours to process (None = all tables since last dump generation)"
        default_value = None
        required = False

    type = "ONLINE_LEARNING_DUMP_TXT_PREPARE"

    input_parameters = [
        MLTaskID,
        TaskParamsConf,
        TaskHoursBefore,
        YTTokenOwner,
        YTTokenName,
        YQLTokenOwner,
        YQLTokenName,
        YTProxy,
        Stable,
        FixedBinaryID,
        MaxDelayHours
    ]

    def on_execute(self):

        output_feature_map_filename = "feature_map.txt"
        output_dump_txt_filename = "dump.txt"
        task_id = self.ctx[self.MLTaskID.name]
        logging.info("Start graph launcher for task_id {task_id}".format(task_id=task_id))
        is_stable = self.ctx[self.Stable.name]
        release_status = sdk_sandboxapi.RELEASE_STABLE if is_stable else sdk_sandboxapi.RELEASE_TESTING
        logging.info('Release status: {release_status}'.format(release_status=release_status))
        fixed_resource_id = self.ctx[self.FixedBinaryID.name]
        if fixed_resource_id is not None:
            launcher_binary_resource_id = fixed_resource_id
        else:
            launcher_binary_resource_id = get_last_released_resource(
                resource_types.ONLINE_GRAPH_LAUNCHER_CREATE_DUMP_TXT_BINARY,
                release_status
            ).id

        launcher_binary_path = self.sync_resource(launcher_binary_resource_id)
        yt_token = self.get_vault_data(
            self.ctx[self.YTTokenOwner.name],
            self.ctx[self.YTTokenName.name],
        )
        yql_token = self.get_vault_data(
            self.ctx[self.YQLTokenOwner.name],
            self.ctx[self.YQLTokenName.name],
        )
        yt_proxy = self.ctx[self.YTProxy.name]
        env = os.environ.copy()
        env['YT_TOKEN'] = yt_token
        env['YQL_TOKEN'] = yql_token
        env['YT_PROXY'] = yt_proxy
        logging.info("Working with launcher binary: {launcher_binary}".format(launcher_binary=launcher_binary_path))

        task_filename = self.sync_resource(
            self.ctx[self.TaskParamsConf.name]
        )

        old_feature_map_resource_id = apihelpers.get_last_resource_with_attrs(
            resource_type=resource_types.ONLINE_LEARNING_FEATURE_MAP,
            attrs={"ml_task_id": task_id},
            all_attrs=True
        )

        binary_dump_resource_id = apihelpers.get_last_resource_with_attrs(
            resource_type=resource_types.LM_DUMP_TSV,
            attrs={"task_id": task_id},
            all_attrs=True
        )

        if binary_dump_resource_id:
            old_binary_dump_file = self.sync_resource(binary_dump_resource_id)
            binary_dump_last_log_time = binary_dump_resource_id.attributes['log_time']
            logging.info("Binary dump last_log_time: {last_log_time}".format(last_log_time=binary_dump_last_log_time))
        else:
            return

        old_feature_map_file = 'None'
        if old_feature_map_resource_id:
            old_feature_map_file = self.sync_resource(old_feature_map_resource_id)
            feature_map_last_log_time = old_feature_map_resource_id.attributes['log_time']
            logging.info("Feature map last_log_time: {last_log_time}".format(last_log_time=feature_map_last_log_time))

            max_delay = self.ctx[self.MaxDelayHours.name]
            if max_delay is not None and max_delay > 0:
                binary_dump_datetime = datetime.datetime.strptime(binary_dump_last_log_time, DATETIME_FORMAT)
                feature_map_datetime = datetime.datetime.strptime(feature_map_last_log_time, DATETIME_FORMAT)
                feature_map_last_log_time = max(binary_dump_datetime - datetime.timedelta(hours=max_delay), feature_map_datetime).strftime(DATETIME_FORMAT)

            if feature_map_last_log_time >= binary_dump_last_log_time:
                # When feature_map_last_log_time >= binary_dump_last_log_time we had no new dumps
                # Or occasionaly production dump fresher
                if feature_map_last_log_time > binary_dump_last_log_time:
                    logging.warning(
                        "Feature map last_log_time:{feature_map_last_log_time} >  "
                        "binary_dump_last_log_time: {binary_dump_last_log_time}".format(
                            feature_map_last_log_time=feature_map_last_log_time,
                            binary_dump_last_log_time=binary_dump_last_log_time
                        )
                    )
                return
        else:
            # When we make first launch without feature map, we take last 6 tables before binary dump
            hours_before = self.ctx[self.TaskHoursBefore.name]
            binary_dump_datetime = datetime.datetime.strptime(binary_dump_last_log_time, DATETIME_FORMAT)
            feature_map_last_log_datetime = binary_dump_datetime - datetime.timedelta(hours=hours_before)
            feature_map_last_log_time = feature_map_last_log_datetime.strftime(DATETIME_FORMAT)

        command = [
            launcher_binary_path,
            "--task", task_filename,
            "--old_feature_map", old_feature_map_file,
            "--binary_dump", old_binary_dump_file,
            "--output_feature_map_file_name", output_feature_map_filename,
            "--output_dump_txt_filename", output_dump_txt_filename,
            "--feature_map_last_log_time", feature_map_last_log_time,
            "--binary_dump_last_log_time", binary_dump_last_log_time
        ]

        logging.info("Start with command {command}".format(command=command))

        ###############################################################################
        graphite_hosts = YABS_SERVERS
        graphite_obj = Graphite(graphite_hosts, datetime.timedelta(seconds=5).total_seconds())

        ################################################################################
        start_time = datetime.datetime.now()
        sdk_process.run_process(
            command,
            wait=True,
            log_prefix='run_online_launcher',
            environment=env
        )

        logging.info(
            "Finish processing {task_id} feature_map saved in file {output_feature_map_filename}".format(
                task_id=task_id,
                output_feature_map_filename=output_feature_map_filename
            )
        )

        end_time = datetime.datetime.now()
        delta = end_time - start_time
        value = delta.total_seconds()
        metric_prefix = five_min_metric(
            "online_learning.graph_runtime_info.task_id:{task_id}.dump_txt_work_time".format(task_id=task_id)
        )
        binary_dump_datetime = datetime.datetime.strptime(binary_dump_last_log_time, DATETIME_FORMAT)
        feature_map_last_log_datetime = datetime.datetime.strptime(feature_map_last_log_time, DATETIME_FORMAT)
        graphite_obj.send([(metric_prefix, value, time.time())])
        delta_time = (binary_dump_datetime - feature_map_last_log_datetime)
        n_dumps = int(delta_time.total_seconds()/3600)
        metric_prefix = five_min_metric(
            "online_learning.graph_runtime_info.task_id:{task_id}.dump_txt_work_time_per_graph".format(task_id=task_id)
        )
        graphite_obj.send([(metric_prefix, int(float(value)/n_dumps), time.time())])

        save_state(
            self,
            output_feature_map_filename,
            resource_types.ONLINE_LEARNING_FEATURE_MAP,
            self.owner,
            last_log_time=binary_dump_last_log_time
        )

        save_state(
            self,
            output_dump_txt_filename,
            resource_types.ONLINE_LEARNING_DUMP_TXT,
            self.owner,
            last_log_time=binary_dump_last_log_time
        )


__Task__ = OnlineLearningDumpTXTPrepare
