# coding: utf-8
import logging
import os
import re
import shutil
import urllib2

import sandbox.common.types.misc  as ctm
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask


class JavaVersionParam(SandboxStringParameter):
    name = 'java_version'
    description = 'Java version'
    required = True
    url = 'http://download.oracle.com/otn-pub/java/jdk/{version}u{update}-{build}/jdk-{version}u{update}-linux-x64.tar.gz'
    default_value = '1.8.0_74-b02'


class JavaUrlParam(SandboxStringParameter):
    name = 'java_url'
    description = 'Java url'
    required = False
    default_value = 'http://download.oracle.com/otn-pub/java/jdk/8u144-b01/090f390dda5b47b9b721c7dfaa008135/jdk-8u144-linux-x64.tar.gz'


class CreateJavaJDK(SandboxTask, nanny.ReleaseToNannyTask):
    """
    Download java from Orcale and create resource
    """
    type = 'CREATE_JAVA_JDK'
    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = (JavaVersionParam, JavaUrlParam)

    execution_space = 1000

    def on_execute(self):
        version = re.match(r'^\d.(?P<version>\d).+_(?P<update>\d+)-(?P<build>.+)$',
                           self.ctx[JavaVersionParam.name]).groupdict()
        self.descr = "java " + self.ctx[JavaVersionParam.name]
        url = self.ctx[JavaUrlParam.name] or JavaVersionParam.url.format(**version)
        logging.info('Downloading: ' + url)
        download_path = self.path('jdk.tar.gz')
        download_file(url, download_path)
        java_path = make_folder('java', True)
        run_process(
            ['tar', '--strip-components=1', '-xvf', download_path, '-C', java_path],
            'tar',
            shell=True
        )
        cert_file = 'yandexinternalrootca.crt'
        run_process(['curl', 'https://crls.yandex.net/YandexInternalRootCA.crt', '>', cert_file], 'curl', shell=True)
        keytool = os.path.join(java_path, 'bin/keytool')
        run_process([keytool, '-printcert', '-file', cert_file], 'keytool', shell=True)
        run_process([keytool, '-noprompt', '-import', '-v', '-trustcacerts', '-alias', 'yandexinternalrootca',
                     '-file', cert_file, '-storepass', 'changeit', '-keystore',
                     os.path.join(java_path, 'jre/lib/security/cacerts')], 'keytool', shell=True)
        resource = self.create_resource(
            self.ctx[JavaVersionParam.name], java_path, resource_types.VPS_JDK_DIST,
            self.client_info['arch'],
            attributes={
                'version': self.ctx[JavaVersionParam.name],
                'platform': self.client_info['platform']
            }
        )
        self.mark_resource_ready(resource)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)


def download_file(url, target):
    url = urllib2.Request(url, headers={'Cookie': 'oraclelicense=accept-securebackup-cookie'})
    url = urllib2.urlopen(url)
    with open(target, 'wb') as f:
        shutil.copyfileobj(url, f)


__Task__ = CreateJavaJDK
