"""Common interfaces for creating packages.
"""

import logging
import os

import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common.gnupg import GpgKey

from sandbox.projects.CreatePackage import deb
from sandbox.projects.CreatePackage import pkg
from sandbox.projects.CreatePackage import upload
from sandbox.projects.CreatePackage import util


EXAMPLE_PKG_SVN_URL = \
    'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/packages/your_package'

DEFAULT_PKG_VERSION = '1.0'
DEFAULT_REPO_NAME = 'search'
DEFAULT_REPO_KEY_ID = 'Full Name <user@domain.tld>'
DEFAULT_GPG_PRIVATE_KEY = 'gpg-private-key'
DEFAULT_GPG_PUBLIC_KEY = 'gpg-public-key'
DEFAULT_SSH_PRIVATE_KEY = 'ssh-private-key'

ROOT_DIR = 'root'


class CreatePackage(SandboxTask):
    """Base class for creating packages.
    """

    class PkgSvnUrl(sdk_parameters.SandboxSvnUrlParameter):
        name = 'pkg_svn_url'
        description = 'Package svn url'
        default_value = EXAMPLE_PKG_SVN_URL

    class PkgVersion(sdk_parameters.SandboxStringParameter):
        name = 'pkg_version'
        description = 'New package version'
        default_value = DEFAULT_PKG_VERSION

    class ResourceIds(sdk_parameters.ResourceSelector):
        multiple = True
        name = 'resource_ids'
        description = 'Resource(s) with package files'

    class UploadToRepo(sdk_parameters.SandboxRadioParameter):
        name = 'upload_to_repo'
        description = 'Upload package to repository'
        choices = [
            ('enabled', 'enabled'),
            ('disabled', 'disabled'),
        ]
        per_line = 2
        default_value = 'enabled'

    class RepoName(sdk_parameters.SandboxStringParameter):
        name = 'repo_name'
        description = 'Repo name'
        default_value = DEFAULT_REPO_NAME

    class RepoKeyId(sdk_parameters.SandboxStringParameter):
        name = 'repo_key_id'
        description = 'Repo key ID (for gpg/ssh)'
        default_value = DEFAULT_REPO_KEY_ID

    class InfoKeys(sdk_parameters.SandboxInfoParameter):
        description = 'Keys from SandboxVault'

    class GpgPrivateKey(sdk_parameters.SandboxStringParameter):
        name = 'gpg_private_key'
        description = 'GPG private key name'
        default_value = DEFAULT_GPG_PRIVATE_KEY

    class GpgPublicKey(sdk_parameters.SandboxStringParameter):
        name = 'gpg_public_key'
        description = 'GPG public key name'
        default_value = DEFAULT_GPG_PUBLIC_KEY

    class SshPrivateKey(sdk_parameters.SandboxStringParameter):
        name = 'ssh_private_key'
        description = 'SSH private key name'
        default_value = DEFAULT_SSH_PRIVATE_KEY

    UploadToRepo.sub_fields = {
        'enabled': [
            RepoName.name,
            RepoKeyId.name,
            InfoKeys.name,
            GpgPrivateKey.name,
            GpgPublicKey.name,
            SshPrivateKey.name,
        ],
        'disabled': [],
    }

    type = 'CREATE_PACKAGE'
    input_parameters = [
        PkgSvnUrl, PkgVersion, ResourceIds,
        UploadToRepo, RepoName, RepoKeyId, InfoKeys, GpgPrivateKey, GpgPublicKey, SshPrivateKey
    ]

    release_to = ['rdna@yandex-team.ru', ]
    execution_space = 20 << 10  # MB

    def on_execute(self):
        raise NotImplementedError

    def create_freebsd_package(self):
        """Creates FreeBSD packages.
        :return: Nothing.
        """
        pkg_meta_dir = util.get_svn_url(self.ctx['pkg_svn_url'])
        pkg_manifest = pkg.Manifest(pkg_meta_dir)
        pkg_manifest.load()

        pkg_manifest.set_version(self.ctx['pkg_version'])

        pkg_root_dir = self.abs_path(ROOT_DIR)
        util.mk_tree(pkg_root_dir, pkg_manifest.get_dirs())

        resources_files = self.get_resources_files()
        util.install_files(resources_files, pkg_root_dir,
                           pkg_manifest.get_files())
        pkg_manifest.set_flatsize(pkg_root_dir)

        pkg.create(pkg_root_dir, pkg_meta_dir)
        pkg_archive = pkg_manifest.get_archive()
        if self.ctx['upload_to_repo'] == 'enabled':
            # TODO(rdna@): Support repo_name for FreeBSD packages.
            upload.run(pkg_archive)

        self.ctx['pkg_full_name'] = pkg_manifest.get_full_name()

    def create_linux_package(self):
        """Creates Linux packages.
        :return: Nothing.
        """
        deb_work_dir = util.get_svn_url(self.ctx['pkg_svn_url'])
        deb_pkg = deb.Package(deb_work_dir)

        # One debian source package may contain more than one binary packages.
        deb_pkg.set_names()
        deb_pkg.set_version(self.ctx['pkg_version'])
        deb_pkg.set_install_files()

        resources_files = self.get_resources_files()
        util.install_files(
            resources_files, deb_work_dir,
            deb_pkg.get_missing_files(), is_rel=True
        )

        if self.is_backward_compatibility_enabled():
            logging.info('Backward compatibility mode.')
            deb_pkg.mk_changelog()
            deb_pkg.build()
            if self.ctx['upload_to_repo'] == 'enabled':
                deb_pkg.upload(self.ctx['repo_name'])
        else:
            with GpgKey(self, self.owner, self.ctx['gpg_private_key'],
                        self.ctx['gpg_public_key']):
                deb_pkg.mk_changelog(self.ctx['repo_key_id'])

                # add log to catch keys problem
                run_process(['gpg', '--list-secret-keys'], log_prefix='gpg_secret_keys')

                deb_pkg.build(self.ctx['repo_key_id'])
                if self.ctx['upload_to_repo'] == 'enabled':
                    with ssh.Key(self, self.owner, self.ctx['ssh_private_key']):
                        ssh_login = util.get_login_from_email(self.ctx['repo_key_id'])
                        deb_pkg.upload(self.ctx['repo_name'], ssh_login)

        self.ctx['pkg_full_name'] = deb_pkg.get_full_name()

    def is_backward_compatibility_enabled(self):
        return self.ctx['repo_key_id'] == DEFAULT_REPO_KEY_ID

    def get_resources_files(self):
        """Gets all resources which are set via form and makes a dictionary
        with all the files from the resources.
        :return: Dictionary: file name -> file path.
        """
        resources_files = {}
        for resource_id in self.ctx['resource_ids']:
            logging.info('Get resource id %s.' % resource_id)
            self.sync_resource(resource_id)
            resource = channel.sandbox.get_resource(resource_id)
            if os.path.isfile(resource.path):
                filename = os.path.basename(resource.path)
                resources_files[filename] = resource.path
            else:
                for dirpath, _, filenames in os.walk(resource.path,
                                                     onerror=True):
                    for filename in filenames:
                        resources_files[filename] = os.path.join(dirpath,
                                                                 filename)
        logging.info('Files in resource(s): %s.' % resources_files)
        return resources_files

# vim:set ts=4 sw=4 et:


__Task__ = CreatePackage
