"""Interface to work with Debian packages.

IMPORTANT
  One debian source package may contain more than one binary packages.
"""

import logging
import os
import subprocess

from sandbox.projects.common.debpkg import DebRelease

from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail
from sandbox.sandboxsdk.process import run_process


DEBIAN_DIR = 'debian'

CHANGELOG_FILE = os.path.join(DEBIAN_DIR, 'changelog')
CHANGELOG_TEXT = 'Created by sandbox task.'

INSTALL_FILE_SFX = '.install'

DCH_BIN = 'dch'
DCH_OPTS = (
    '--create --package %(package)s --newversion %(version)s '
    '--force-distribution -D unstable %(text)s'
)

DEBUILD_BIN = 'debuild'
DEBUILD_OPTS = '-b'

DEBRELEASE_OPTS = '--to %(repo_name)s'

DEFAULT_DIST_USER = 'sandbox'

DUPLOAD_CONF_TPL = {
    'search': {
        'fqdn': 'search.dupload.dist.yandex.ru',
        'method': 'scpb',
        'login': DEFAULT_DIST_USER,
        'incoming': '/repo/search/mini-dinstall/incoming/',
        'dinstall_runs': 1,
    },
    'search-precise': {
        'fqdn': 'search-precise.dupload.dist.yandex.ru',
        'method': 'scpb',
        'login': DEFAULT_DIST_USER,
        'incoming': '/repo/search-precise/mini-dinstall/incoming/',
        'dinstall_runs': 1,
    },
    'yandex-lucid': {
        'fqdn': 'yandex-lucid.dupload.dist.yandex.ru',
        'method': 'scpb',
        'login': DEFAULT_DIST_USER,
        'incoming': '/repo/yandex-lucid/mini-dinstall/incoming/',
        'dinstall_runs': 1,
    },
    'yandex-precise': {
        'fqdn': 'yandex-precise.dupload.dist.yandex.ru',
        'method': 'scpb',
        'login': DEFAULT_DIST_USER,
        'incoming': '/repo/yandex-precise/mini-dinstall/incoming/',
        'dinstall_runs': 1,
    },
    'yandex-trusty': {
        'fqdn': 'yandex-trusty.dupload.dist.yandex.ru',
        'method': 'scpb',
        'login': DEFAULT_DIST_USER,
        'incoming': '/repo/yandex-trusty/mini-dinstall/incoming/',
        'dinstall_runs': 1,
    },
}


class Package(object):
    """Class to work with Debian packages.
    """
    # TODO(rdna@): Avoid __acquire_dir() / __release_dir() functions.

    def __init__(self, work_dir, key_id=None):
        self.work_dir = work_dir
        self.changelog_file = os.path.join(self.work_dir, CHANGELOG_FILE)

    def __acquire_dir(self):
        """Remembers current directory and goes to working directory of
        package.
        :return: Nothing.
        """
        self.save_dir = os.getcwd()
        os.chdir(self.work_dir)

    def __release_dir(self):
        """Comes back to saved directory.
        :return: Nothing.
        """
        os.chdir(self.save_dir)

    def set_names(self):
        """Gets name(s) of current binary package(s) and sets appropriate class
        members.
        :return: Nothing.
        """
        logging.info('Set package name(s).')
        self.__acquire_dir()
        self.names = subprocess.check_output('dh_listpackages').split()
        self.__release_dir()
        self.main_name = self.names[0]
        logging.info('Main package name is %s.' % self.main_name)
        if len(self.names) > 1:
            logging.info('Additional package names are %s.' % self.names[1:])

    def set_version(self, version):
        """Sets version of current package to @version.
        :return: Nothing.
        """
        logging.info('Set package version to %s.' % version)
        self.version = version

    def set_install_files(self):
        """Finds out path(s) to install file(s) of current package(s) and sets
        appropriate class member to it.
        :return: Nothing.
        """
        self.install_files = set()
        for name in self.names:
            file_name = name + INSTALL_FILE_SFX
            install_file = os.path.join(self.work_dir, DEBIAN_DIR, file_name)
            if os.path.isfile(install_file):
                self.install_files.add(install_file)
        logging.info('Install file(s): %s.' % self.install_files)

    def mk_changelog(self, debemail=None):
        """Makes new changelog file for package. It doesn't save previous
        changelog entries nor commit them.
        :return: Nothing.
        """
        logging.info('Make changelog file.')
        if os.path.isfile(self.changelog_file):
            os.unlink(self.changelog_file)
        dch_opts = DCH_OPTS % {
            'package': self.main_name,
            'version': self.version,
            'text': CHANGELOG_TEXT,
        }
        dch_cmd = '%s %s' % (DCH_BIN, dch_opts)
        env = os.environ.copy()
        if debemail:
            env['DEBEMAIL'] = debemail
        log_prefix = os.path.basename(DCH_BIN)
        run_process(
            dch_cmd, environment=env, work_dir=self.work_dir,
            log_prefix=log_prefix
        )
        with open(self.changelog_file) as chlog_fd:
            chlog_str = chlog_fd.read()
        logging.info('Changelog file is created:\n%s' % chlog_str)

    def __get_src_files(self):
        """Parses install file(s) of current package to get all source files
        necessary for building.
        :return: Set of source files.
        """
        src_files = set()
        for install_file in self.install_files:
            with open(install_file) as install_fd:
                for line in install_fd.readlines():
                    # Each line lists a file or files to install, and at the
                    # end of the line tells the directory it should be
                    # installed in. See dh_install(1) for details.
                    src_files.update(line.split()[:-1])
        logging.info('Source files: %s.' % src_files)
        return src_files

    def get_full_name(self):
        """Getter for package full main name.
        :return: Package full main name in format suitable for using with
                 ``apt-get(8) install''.
        """
        return '%s=%s' % (self.main_name, self.version)

    def get_missing_files(self):
        """Checks list of src files of current package and gets those which are
        not presented in package working directory now.
        :return: Set of missing files.
        """
        missing_files = set()
        for src_file in self.__get_src_files():
            src_path = os.path.join(self.work_dir, src_file)
            if not os.path.exists(src_path):
                missing_files.add(src_file)
        return missing_files

    def build(self, key_id=None, preserve_env=False):
        """Builds current package.
        :return: Nothing.
        """
        logging.info('Build Debian package.')
        build_cmd_str = '%s %s' % (DEBUILD_BIN, DEBUILD_OPTS)
        build_cmd = build_cmd_str.split()
        if key_id:
            build_cmd += ['-k%s' % key_id]
        build_cmd += ['-j16']
        # --no-lintian is not understood I don't know why
        # build_cmd += ['-j16', '--no-lintian']
        if preserve_env:
            build_cmd[1:1] = ['--preserve-env', '--preserve-envvar', 'PATH']
        log_prefix = os.path.basename(DEBUILD_BIN)
        run_process(build_cmd, work_dir=self.work_dir, log_prefix=log_prefix)

    def upload(self, repo_name, login=None):
        """Uploads built package to package repository @repo_name.
        :return: Nothing.
        """
        logging.info('Upload Debian package to %s repo.' % repo_name)

        if repo_name not in DUPLOAD_CONF_TPL:
            raise TaskFail('Unsupported repo: %s.' % repo_name)
        repo_props = DUPLOAD_CONF_TPL[repo_name]

        if login:
            repo_props['login'] = login

        dupload_conf = {repo_name: repo_props}
        debrelease_opts = DEBRELEASE_OPTS % {'repo_name': repo_name}

        self.__acquire_dir()
        with DebRelease(dupload_conf) as deb:
            deb.debrelease(debrelease_opts.split())
        self.__release_dir()
