"""Interface to work with FreeBSD package manager pkg(8).
WWW: https://wiki.freebsd.org/pkgng
"""

import logging
import os
import re
import yaml

from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail
from sandbox.sandboxsdk.process import run_process


MANIFEST_FILE = '+MANIFEST'

PKG_BIN = '/usr/local/sbin/pkg'
PKG_FORMAT = 'txz'


class Manifest(object):
    """Class to work with pkg(8) manifest files.
    WWW: https://wiki.freebsd.org/pkgng#Metadata
    """

    def __init__(self, meta_dir):
        self.m_path = os.path.join(meta_dir, MANIFEST_FILE)

    def load(self):
        """Loads data from manifest file which is pointed by @m_path class
        member and makes @m_dict class member with this data.
        :return: Nothing.
        """
        logging.info('Load manifest from %s.' % self.m_path)
        with open(self.m_path) as m_fd:
            self.m_str = m_fd.read()

        # TODO(rdna@):
        # YAML converts integers from octal to decimal and therefore breaks
        # ``perm'' nodes. This workaround converts ``perm'' from integer to
        # string before loading by YAML. We should learn how to do using just
        # YAML.
        self.m_str = re.sub(r'(\W??perm:\s*)(0[0-7]{3})(\D??)',
                            r'\1"\2"\3',
                            self.m_str)

        self.m_dict = yaml.load(self.m_str)
        logging.info('Manifest contains: %s.' % self.m_dict)

    def dump(self):
        """Dumps manifest file pointed to by @m_path class member to disk.
        :return: Nothing.
        """
        self.m_str = yaml.dump(self.m_dict)
        with open(self.m_path, 'w') as m_fd:
            m_fd.write(self.m_str)

    def set_version(self, version):
        """Sets package version in current @m_dict class member to @version.
        :return: Nothing.
        """
        logging.info('Set version "%s" for package %s.' % (version,
                                                           self.m_dict['name']))
        self.m_dict['version'] = version
        self.dump()

    def set_flatsize(self, root_dir):
        """Computes flatsize for all the files in @root_dir and sets package
        flatsize in current @m_dict class member to it
        :return: Nothing.
        """
        flatsize = 0
        for dirpath, _, filenames in os.walk(root_dir, onerror=True):
            for filename in filenames:
                filepath = os.path.join(dirpath, filename)
                flatsize += os.stat(filepath).st_size
        self.m_dict['flatsize'] = flatsize
        self.dump()

    def get_path(self):
        """Getter for @m_path class member.
        :return: Path to current manifest file.
        """
        return self.m_path

    def get_dirs(self):
        """Getter for ``dirs'' and ``directories'' from current manifest file.
        :return: Set of directory names.
        """
        dirs = set()
        if self.m_dict.get('directories'):
            dirs.update(self.m_dict['directories'].keys())
        if self.m_dict.get('dirs'):
            for dir_dict in self.m_dict['dirs']:
                dirs.update(dir_dict.keys())
        return dirs

    def get_files(self):
        """Getter for ``files'' from current manifest file.
        :return: Set of file names.
        """
        return set(self.m_dict['files'].keys())

    def get_archive(self):
        """Constructs package archive name using values from current manifest
        file.
        :return: Package archive name.
        """
        pkg_name = self.m_dict['name']
        pkg_version = self.m_dict['version']
        pkg_archive = '%s-%s.%s' % (pkg_name, pkg_version, PKG_FORMAT)
        if not os.path.isfile(pkg_archive):
            raise TaskFail('Archive %s does not exist.' % pkg_archive)
        logging.info('Package archive: %s.' % pkg_archive)
        return pkg_archive

    def get_full_name(self):
        """Getter for package full name.
        :return: Package full name in format suitable for using with
                 ``pkg(8) install''.
        """
        return '%(name)s-%(version)s' % self.m_dict


def create(root_dir, manifest_dir):
    """Creates package using pkg-create(8), files from @root_dir and meta-data
    from @manifest_dir.
    :return: Nothing.
    """
    pkg_create_cmd = '%s create -m %s -r %s' % (PKG_BIN, manifest_dir, root_dir)
    run_process(pkg_create_cmd, log_prefix=os.path.basename(PKG_BIN))
