"""Interface to upload package archives onto package server.
"""

import os
import socket

from kernel.util.functional import reliable

from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.CreatePackage import util


EUploadMode = util.enum(
    UM_DEV=1,
    UM_PROD=2,
    )

EUploadUser = util.enum(
    UU_DEV='rdna',
    UU_PROD='sandbox',
    )

EUploadPassFile = util.enum(
    PF_DEV='/home/%(upload_user)s/.rsync_pass',
    PF_PROD='/usr/local/etc/rsync/secret.%(upload_user)s',
    )

UPLOAD_CMD = 'rsync'
UPLOAD_DST = 'rsync://%(upload_user)s@dst02.yandex.ru/sandbox'
UPLOAD_OPTS = '--contimeout=30 --timeout=120 -tv ' + \
    '--password-file=%(upload_pass_file)s --temp-dir=tmp'
UPLOAD_TRIES = 5            # times
UPLOAD_TRY_INTERVAL = 15    # seconds


def get_upload_mode():
    """Gets upload mode according to FQDN of current host.
    :return: Enumerator from EUploadMode enum.
    """
    return {
        'ed209.yandex.ru': EUploadMode.UM_DEV,
        'manatus.yandex.ru': EUploadMode.UM_DEV,
        }.get(socket.getfqdn(), EUploadMode.UM_PROD)


def get_upload_user(upload_mode):
    """Gets upload user according to current @upload_mode.
    :return: Enumerator from EUploadUser enum.
    """
    return {
        EUploadMode.UM_DEV: EUploadUser.UU_DEV,
        EUploadMode.UM_PROD: EUploadUser.UU_PROD,
        }[upload_mode]


def get_upload_pass_file(upload_mode, upload_user):
    """Gets upload password file according to @upload_mode and @upload_user.
    :return: Enumerator from EUploadPassFile enum with evaluated
    ``upload_user'' variable.
    """
    return {
        EUploadMode.UM_DEV: EUploadPassFile.PF_DEV,
        EUploadMode.UM_PROD: EUploadPassFile.PF_PROD,
        }[upload_mode] % {'upload_user': upload_user}


@reliable(UPLOAD_TRIES, SandboxSubprocessError, UPLOAD_TRY_INTERVAL)
def run_upload_cmd(upload_cmd):
    """Runs @upload_cmd reliably using appropriate decorator.
    :return: Nothing.
    """
    run_process(upload_cmd, log_prefix=os.path.basename(UPLOAD_CMD))


def run(pkg_archive):
    """Uploads package archives @pkg_archive to package server.
    :return: Nothing.
    """
    upload_mode = get_upload_mode()
    upload_user = get_upload_user(upload_mode)
    upload_pass_file = get_upload_pass_file(upload_mode, upload_user)

    upload_dst = UPLOAD_DST % {'upload_user': upload_user}
    upload_opts = UPLOAD_OPTS % {'upload_pass_file': upload_pass_file}

    upload_cmd = '%(cmd)s %(opts)s %(src)s %(dst)s' % {
        'cmd': UPLOAD_CMD,
        'opts': upload_opts,
        'src': pkg_archive,
        'dst': upload_dst,
        }
    run_upload_cmd(upload_cmd)
