"""Utility functions."""

import logging
import os
import shutil

from email.utils import parseaddr

from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskFail
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.svn import Arcadia


def enum(**enums):
    """Simulates enum from C. Creates a new class where keys from @enums
    dictionary became class variables with values equal values from @enums.
    :return: Created class.
    """
    return type('Enum', (object, ), enums)


def get_svn_url(svn_url, isfile=False):
    """Gets @svn_url from subversion server to local server. URL is treated as
    a file if @isfile is True and as a directory otherwise.
    :return: Local path.
    """
    logging.info('Get svn url %s.' % svn_url)
    local_path = os.path.abspath(os.path.basename(svn_url))
    if isfile:
        Arcadia.export(svn_url, local_path)
        return local_path
    else:
        return Arcadia.checkout(svn_url, local_path)


def mk_tree(root_dir, sub_dirs):
    """Makes sub-directories @sub_dirs in @root_dir.
    :return: Nothing.
    """
    make_folder(root_dir)
    for sub_dir in sub_dirs:
        rel_dir = os.path.relpath(sub_dir, start='/')
        abs_dir = os.path.join(root_dir, rel_dir)
        make_folder(abs_dir)


def install_files(all_files, pkg_root_dir, pkg_files, is_rel=False):
    """Installs files from @all_files as @pkg_files to directory @pkg_root_dir.
    Files @pkg_files are treated as absolute paths if @is_rel is False and as
    relative ones otherwise.
    @pkg_files consume just subset of @all_files because handled resources may
    contain something that should not be added to the package.
    :return: Nothing.
    """
    if is_rel:
        start = '.'
    else:
        start = '/'
    for pkg_file in pkg_files:
        pkg_rel_file = os.path.relpath(pkg_file, start=start)
        pkg_filename = os.path.basename(pkg_file)
        if not all_files.get(pkg_filename):
            raise TaskFail('Where to get %s ?' % pkg_file)
        pkg_src_file = all_files[pkg_filename]
        pkg_dst_file = os.path.join(pkg_root_dir, pkg_rel_file)
        pkg_dst_dir = os.path.dirname(pkg_dst_file)
        if not os.path.exists(pkg_dst_dir):
            os.makedirs(pkg_dst_dir)
        shutil.copy(pkg_src_file, pkg_dst_file)


def get_login_from_email(email):
    """Gets login from ``Full Name <login@domain.tld>.
    :return: Login string.
    """
    (real_name, email_address) = parseaddr(email)
    return email_address.split('@')[0]
