# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters
from sandbox.projects.common import utils
from sandbox.projects.common import string


def _create_text_res_params(group_name=None):
    class ParamsClass(object):
        class ResName(parameters.SandboxStringParameter):
            name = 'created_resource_name'
            description = 'Created resource name'
            default_value = 'uploaded_resource'
            required = True
            group = group_name

        class FileContent(parameters.SandboxStringParameter):
            name = 'resource_file_content'
            description = 'Resource file content'
            multiline = True
            required = True
            group = group_name

        class DoNotRemove(parameters.SandboxBoolParameter):
            name = 'store_forever'
            description = 'Store resource forever (assign TTL to infinity)'
            do_not_copy = True
            default_value = False
            group = group_name

        class ResAttrs(parameters.SandboxStringParameter):
            name = 'resource_attrs'
            description = 'Set attrs to resources (e.g.: attr1=v1, attr2=v2)'
            do_not_copy = True
            group = group_name

        params = [
            ResName,
            FileContent,
            DoNotRemove,
            ResAttrs,
        ]

    return ParamsClass


class CreateTextResource(SandboxTask):
    """
        создает ресурс путем указания типа и заполнения многострочного текстового поля
    """

    type = 'CREATE_TEXT_RESOURCE'

    cores = 1

    input_parameters = utils.get_resource_type_params() + _create_text_res_params().params

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        param_name = 'resource_arch'
        resource_arch = self.ctx.get(param_name, 'any')

        self.ctx['out_resource_id'] = self.create_resource(
            self.descr,
            self.ctx['created_resource_name'],
            self.ctx['resource_type'],
            arch=resource_arch
        ).id

    def on_execute(self):
        resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        resource_content = self.ctx['resource_file_content'].replace('\r\n', '\n').encode('utf-8')
        with open(resource.path, "w") as file:
            file.write(resource_content)

        del self.ctx['resource_file_content']

        resource_attributes = self.ctx.get('resource_attrs')
        if resource_attributes:
            for k, v in string.parse_attrs(resource_attributes).iteritems():
                channel.sandbox.set_resource_attribute(resource.id, k, v)

        if self.ctx.get('store_forever', False):
            channel.sandbox.set_resource_attribute(resource.id, 'ttl', 'inf')


__Task__ = CreateTextResource
