__author__ = 'bykanov'

import logging
import json

from sandbox import common
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.channel as sdk_channel


application_parameters_group = 'Delete parameters'


class DeleteOldResources(sdk_task.SandboxTask):
    """ DANGER! Run it only if you know what you doing.
        This task deletes old sandbox resources """

    class ResourcesOwner(sdk_parameters.SandboxStringParameter):
        name = 'resources_owner'
        required = True
        description = 'Resources owner'
        group = application_parameters_group

    class ChoiceAttribute(sdk_parameters.SandboxStringParameter):
        name = 'choice_attribute'
        required = True
        description = 'Attribute for choosing sandbox resources'
        default_value = 'name'
        group = application_parameters_group

    class AdditionalAttribute(sdk_parameters.SandboxStringParameter):
        name = 'additional_attribute'
        required = True
        description = 'Additional attribute.'
        default_value = '{"autodeploy": "yes"}'
        group = application_parameters_group

    class PreserveResourcesCount(sdk_parameters.SandboxStringParameter):
        name = 'preserve_resources_count'
        required = True
        description = 'Count of preserved resources for each choice attribute value'
        default_value = '10'
        group = application_parameters_group

    class DryRun(sdk_parameters.SandboxBoolParameter):
        name = 'dry_run'
        description = 'Dry run'
        default_value = True
        group = application_parameters_group

    type = 'DELETE_OLD_RESOURCES'
    cores = 1
    input_parameters = [ResourcesOwner, ChoiceAttribute, AdditionalAttribute, PreserveResourcesCount, DryRun]

    def on_execute(self):
        last_resources = self.get_last_sandbox_resources()
        resources_count = {}
        for res in last_resources:
            # Paranoid check self.ctx['choice_attribute'] existance for sandbox resource
            if not self.ctx['choice_attribute'] in res['attributes'] or not res['attributes'][self.ctx['choice_attribute']].strip():
                continue

            res_name = res['attributes'][self.ctx['choice_attribute']].strip()
            resources_count[res_name] = resources_count.get(res_name, 0) + 1

            # Delete resource
            if resources_count[res_name] > int(self.ctx['preserve_resources_count'].strip()) and resources_count[res_name] > 2:
                logging.info('Delete resource: %s ', json.dumps(res))
                if not self.ctx['dry_run']:
                    sdk_channel.channel.sandbox.set_resource_attribute(res['id'], 'ttl', 1)
                    sdk_channel.channel.sandbox.delete_resource(res['id'], ignore_last_usage_time=True)

        logging.debug('resources_count: %s ', json.dumps(resources_count))

    def get_last_sandbox_resources(self):
        sandbox_client = common.rest.Client()

        # Create json for request
        request = {}
        request['state'] = 'READY'
        request['owner'] = self.ctx['resources_owner'].strip()
        request['attrs'] = json.loads(self.ctx['additional_attribute'].strip())

        last_resources = sandbox_client.resource[request, :3000]
        if not last_resources or not last_resources['items']:
            self.log_and_raise('Could not find resources with attribute ' + json.dumps(request['attrs']))
        logging.debug('Found resources: %s ', json.dumps(last_resources))
        return last_resources['items']

    def log_and_raise(self, message):
        logging.error(message)
        raise Exception(message)


__Task__ = DeleteOldResources
