import logging
import json

from sandbox import common
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as parameters
import sandbox.sandboxsdk.channel as sdk_channel

from sandbox.projects.common.utils import get_or_default

application_parameters_group = 'Delete parameters'


class ResourcesOwner(parameters.SandboxStringParameter):
    name = 'resources_owner'
    required = True
    description = 'Resources owner'
    group = application_parameters_group


class GroupAttributes(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'group_attributes'
    required = True
    description = 'Attributes for grouping sandbox resources (e.t key for detect equal resources), type is not attribute but we support it there'
    default_value = ['type', 'mark']
    group = application_parameters_group


class GroupAttributesFallback(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'group_attributes_fallback'
    required = True
    description = 'Secondary attributes for grouping sandbox resources (e.t key for detect equal resources), type is not attribute but we support it there'
    default_value = ['type', 'released']
    group = application_parameters_group


class AdditionalAttribute(parameters.DictRepeater, parameters.SandboxStringParameter):
    name = 'additional_attribute'
    required = True
    description = 'Attributes to filter out affected resources'
    default_value = {"ttl": "inf"}
    group = application_parameters_group


class PreserveResourcesCount(parameters.SandboxIntegerParameter):
    name = 'preserve_resources_count'
    required = True
    description = 'Count of preserved resources for each group'
    default_value = 10
    group = application_parameters_group


class SearchResourcesMaximumDeep(parameters.SandboxIntegerParameter):
    name = 'maximum_deep'
    required = True
    description = 'Deep of search'
    default_value = 150
    group = application_parameters_group


class DryRun(parameters.SandboxBoolParameter):
    name = 'dry_run'
    description = 'Dry run'
    default_value = True
    group = application_parameters_group


class DeleteOldResourcesGeosearch(sdk_task.SandboxTask):
    """ DANGER! Run it only if you know what you doing.
        This task deletes old sandbox resources """

    type = 'DELETE_OLD_RESOURCES_GEO'
    cores = 1
    input_parameters = [ResourcesOwner, GroupAttributes, GroupAttributesFallback, AdditionalAttribute, PreserveResourcesCount, SearchResourcesMaximumDeep, DryRun]

    def on_execute(self):
        dry_run = get_or_default(self.ctx, DryRun)
        preserve_count = get_or_default(self.ctx, PreserveResourcesCount)
        group_atributes = [get_or_default(self.ctx, GroupAttributes), get_or_default(self.ctx, GroupAttributesFallback)]

        last_resources = self._get_last_sandbox_resources()
        logging.info("Obtained resources: %d", len(last_resources))

        real_deleted = 0
        not_tagged = 0
        resources_count = {}
        not_tagged_resources = {}
        for res in last_resources:
            for gr in group_atributes:
                res_key = self._build_key(res, gr)
                if res_key:
                    break

            if not res_key:
                not_tagged_resources[res['type']] = not_tagged_resources.get(res['type'], 0) + 1
                not_tagged += 1
                continue

            resources_count[res_key] = resources_count.get(res_key, 0) + 1

            # Delete resource
            if resources_count[res_key] > preserve_count and resources_count[res_key] > 2:
                logging.debug('Delete resource: %s ', json.dumps(res))
                if not dry_run:
                    sdk_channel.channel.sandbox.drop_resource_attribute(res['id'], 'ttl')
                    sdk_channel.channel.sandbox.delete_resource(res['id'], ignore_last_usage_time=True)
                    real_deleted += 1

        logging.info('resources_count: %s ', json.dumps(resources_count))
        logging.info('deleted: %d ', real_deleted)
        logging.info('could not delete: %d ', not_tagged)
        logging.info('not_tagged_resources %s ', json.dumps(not_tagged_resources))

    def _get_attr(self, res, a):
        if a == 'type':
            return res['type']
        else:
            return res['attributes'].get(a)

    def _build_key(self, res, group_atributes):
        res_list = [self._get_attr(res, a) for a in group_atributes]

        if all(res_list):  # All atributes exist and valid
            return '#'.join(res_list)

        return None  # Any error - ignore resource

    def _get_last_sandbox_resources(self):
        sandbox_client = common.rest.Client()

        # Create json for request
        request = {}
        request['state'] = 'READY'
        request['owner'] = get_or_default(self.ctx, ResourcesOwner).strip().strip('"').strip()
        request['attrs'] = get_or_default(self.ctx, AdditionalAttribute)

        accumulated_resources = []
        preserve_count = get_or_default(self.ctx, SearchResourcesMaximumDeep)
        for o in xrange(preserve_count):
            last_resources = sandbox_client.resource[request, o*1000:1000]
            # if not last_resources or not last_resources['items']:
            #    self.log_and_raise('Could not find resources with attribute ' + json.dumps(request['attrs']))
            # logging.debug('Found resources: %s ', json.dumps(last_resources))

            accumulated_resources += last_resources['items']

        accumulated_resources = dict((p['id'], p) for p in accumulated_resources).values()

        return accumulated_resources

    def log_and_raise(self, message):
        logging.error(message)
        raise Exception(message)


__Task__ = DeleteOldResourcesGeosearch
