import os
import logging
import glob
import json

import sandbox.common.types.client as ctc

from sandbox import sandboxsdk

from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter

import sandbox.projects.common.gnupg
import sandbox.projects.common.debpkg
import sandbox.projects.common.build.parameters as build_params


class VaultOwner(SandboxStringParameter):
    name = "vault_owner"
    description = "Sandbox key vault owner"
    group = "Sandbox Vault Settings"
    default_value = "TEAMCITY"


class DeployUser(SandboxStringParameter):
    name = "deploy_user"
    description = "Deploy user"
    group = "Sandbox Vault Settings"
    default_value = "robot-simcity"


class DeployPackage(SandboxStringParameter):
    name = "deploy_package"
    description = "Deploy Packages (space-separated list)"
    group = "Deployment"
    default_value = "all"


class ScriptPath(SandboxStringParameter):
    name = "script_path"
    description = "Custom build_deploy.bash path"
    group = "Deployment"
    default_value = "kikimr/scripts/build_deploy.bash"


class Slice(SandboxStringParameter):
    name = "slice"
    description = "KiKiMR cluster slice to deploy to"
    group = "Deployment"
    default_value = "KIKIMR_MR_MASTER"


class Config(SandboxStringParameter):
    name = "config"
    description = "Kikimr config directory"
    group = "Deployment"
    default_value = ""


class Z2Key(SandboxStringParameter):
    name = "z2_key"
    description = "z2 key for operation execution"
    group = "Deployment"
    default_value = "e80a2755-7d81-4b8c-bcf7-6b1cad5c5d26"


class PostDeployScript(SandboxStringParameter):
    name = "post_deploy_script"
    description = "Run script after deploy (path in arcadia)"
    group = "Deployment"
    default_value = None


class RunPostDeployScript(SandboxBoolParameter):
    name = "run_post_deploy_script"
    description = "Run script after deploy"
    group = "Deployment"
    default_value = False


class DebugMode(SandboxBoolParameter):
    name = "debug_mode"
    description = "Run post_deploy script only"
    group = "Deployment"
    default_value = False


class DoNotDeploy(SandboxBoolParameter):
    name = "do_not_deploy"
    description = "Do not deploy package, build and publish only"
    group = "Deployment"
    default_value = False


class PublishToMaster(SandboxBoolParameter):
    name = "publish_to_master"
    description = "Publish packages to search instead of search-test repo"
    group = "Deployment"
    default_value = False


class Ymake(SandboxStringParameter):
    name = "local_ymake_path"
    description = "Move ymake to arcadia root before building (empty means - no). Specify path in arcadia."
    group = "Build"
    default_value = os.path.join("kikimr", "driver", "local.ymake.template")


class YaRevision(SandboxStringParameter):
    name = 'ya_revision'
    description = "Update devtools/ya to the revision (don't do that by default)"
    group = "Build"
    default_value = None


class NewDeploySyntax(SandboxBoolParameter):
    name = "new_deploy_syntax"
    description = "Use new build_deploy.bash launch syntax"
    group = "Deployment"
    default_value = False


class FastDebianPackaging(SandboxBoolParameter):
    name = "fast_debian_packaging"
    description = "Fast debian packaging"
    group = "Build"
    default_value = False


class StripPackages(SandboxBoolParameter):
    name = "strip_packages"
    description = "Strip packages and put debug info separately"
    group = "Build"
    default_value = False


class LTO(SandboxBoolParameter):
    name = "lto"
    description = "LTO"
    group = "Build"
    default_value = False


class DistBuild(SandboxBoolParameter):
    name = "dist"
    description = "Use dist build"
    group = "Build"
    default_value = False


DUPLOAD_CONF_SEARCH_TEST = {
    'search-test': {
        "fqdn": "search-test.dupload.dist.yandex.ru",
        "method": "scpb",
        "incoming": "/repo/search-test/mini-dinstall/incoming/",
        "dinstall_runs": 1,
    }
}

DUPLOAD_CONF_SEARCH = {
    'search': {
        'fqdn': "search.dupload.dist.yandex.ru",
        'method': "scpb",
        'incoming': "/repo/search/mini-dinstall/incoming/",
        'dinstall_runs': 1,
    }
}


class DeployKiKiMR(ArcadiaTask):
    """
    Builds and deploys kikimr on a specified slice
    """
    type = 'DEPLOY_KIKIMR'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = build_params.get_arcadia_params() + [
        Slice,
        Config,
        DeployUser,
        VaultOwner,
        Z2Key,
        PublishToMaster,
        DeployPackage,
        PostDeployScript,
        RunPostDeployScript,
        ScriptPath,
        DoNotDeploy,
        DebugMode,
        Ymake,
        YaRevision,
        NewDeploySyntax,
        FastDebianPackaging,
        StripPackages,
        LTO,
        DistBuild,
    ]

    def update_ya(self):
        if self.ctx[YaRevision.name]:
            ya_path = os.path.join(self.arcadia, "devtools", "ya")
            Arcadia.update(ya_path, revision=self.ctx[YaRevision.name])

    def run_packer(self):
        if self.ctx[PublishToMaster.name]:
            DUPLOAD_CONF = DUPLOAD_CONF_SEARCH
            repo = "search"
        else:
            DUPLOAD_CONF = DUPLOAD_CONF_SEARCH_TEST
            repo = "search-test"
        config = DUPLOAD_CONF.copy()
        user = self.ctx.get(DeployUser.name)
        owner = self.ctx[VaultOwner.name]
        config[repo]["login"] = user
        with sandbox.projects.common.gnupg.GpgKey(self, owner, "%s-gpg-private" % user, "%s-gpg-public" % user):
            with sandbox.projects.common.debpkg.DebRelease(config):
                with sandboxsdk.ssh.Key(self, owner, "%s-ssh" % user):
                    environment = os.environ.copy()
                    environment['DEBEMAIL'] = '%s@yandex-team.ru' % user
                    return self.run_build_deploy(environment)

    def get_build_script(self):
        path = os.path.join(
            self.abs_path("arcadia"),
            self.ctx[ScriptPath.name]
        )

        if os.path.exists(path):
            return path
        else:
            raise Exception("Failed to obtain 'build_deploy.bash', not found in %s" % path)

    def get_new_deploy_command(self, packages, api_key, config):
        cmd = [
            self.get_build_script(),
            "-a", api_key,
            "-s", self.path(),
            "-u", self.ctx[DeployUser.name],
            "-i",
            "-v", os.path.join(self.log_path(), "packages_versions.txt"),
        ]

        if config:
            cmd += ["-c", config]

        if not self.ctx[DoNotDeploy.name]:
            slice_name = self.ctx[Slice.name]
            cmd += ["-d", slice_name]
        if self.ctx[PublishToMaster.name]:
            cmd += ["-m"]
        if self.ctx[FastDebianPackaging.name]:
            cmd += ["-f"]
        if self.ctx[StripPackages.name]:
            cmd += ["-x"]
        if self.ctx[LTO.name]:
            cmd += ["-l"]
        if self.ctx[DistBuild.name]:
            cmd += ["-e"]

        cmd.extend(packages)
        return cmd

    def get_old_deploy_command(self, package, api_key, config):
        cmd = [
            self.get_build_script(),
            "-p", package,
            "-a", api_key,
            "-s", self.path(),
            "-c", config,
            "-u", self.ctx[DeployUser.name],
            "-f", self.path("temp_dir"),
        ]

        if not self.ctx[DoNotDeploy.name]:
            slice = self.ctx[Slice.name]
            cmd += ["-d", slice]
        if self.ctx[PublishToMaster.name]:
            cmd += ["-m"]
        return cmd

    def run_build_deploy(self, environment):
        api_key = self.ctx[Z2Key.name]
        config = os.path.join("arcadia", self.ctx[Config.name]) if self.ctx[Config.name] else ""
        packages = self.ctx[DeployPackage.name].strip().split()
        packages_description = dict()

        if self.ctx[NewDeploySyntax.name]:
            cmd = self.get_new_deploy_command(packages, api_key, config)
            self._subprocess(cmd, wait=True, check=True, log_prefix="build_deploy", extra_env=environment)
            packages_description["packages"] = self.get_packages_json()
        else:
            for package in packages:
                cmd = self.get_old_deploy_command(package, api_key, config)
                self._subprocess(cmd, wait=True, check=True, log_prefix="build_deploy_" + package, extra_env=environment)
                packages_description[package] = self.get_packages_json()

        return packages_description

    def get_packages_json(self):
        description_files = []
        paths = [self.path("packages.json"), os.path.join(self.path(), "*/packages.json")]
        for path in paths:
            found = glob.glob(path)
            if found:
                logging.info("FOUND %s" % found)
            else:
                logging.info("FAILED TO FIND %s" % path)

            description_files.extend(found)

        if description_files:
            assert len(description_files) == 1, "Found more than one packages.json: " + ", ".join(description_files)
            with open(description_files[0]) as fp:
                res = json.load(fp)
            return res

    def run_post_deploy_script(self, command):
        parsed_command = command.split(" ", 1)
        if len(parsed_command) == 1:
            script = parsed_command[0]
            args = ""
        else:
            script, args = parsed_command

        if os.path.exists(script):
            if not os.access(script, os.X_OK):
                logging.error("File %s is not marked as executable in svn. Trying to do it locally" % script)
                sandboxsdk.paths.chmod(script, 0o755)
            cmd = [script] + args.split()
            logging.info("Executing %s" % str(cmd))
            with sandboxsdk.ssh.Key(self, self.ctx[VaultOwner.name], "%s-ssh" % self.ctx[DeployUser.name]):
                environment = os.environ.copy()
                environment["YR_LOGIN"] = self.ctx[DeployUser.name]
                self._subprocess(cmd, wait=True, check=True, log_prefix="post_deploy", extra_env=environment)
        else:
            raise Exception("Incorrect post deploy script specified: %s" % script)

    def on_execute(self):
        self.arcadia = self.get_arcadia_src_dir()
        patch = self.ctx.get("arcadia_patch")
        if patch:
            Arcadia.apply_patch(self.arcadia, patch + "\n", self.log_path())
        if self.ctx[Ymake.name]:
            ymake_path = os.path.join(self.arcadia, self.ctx[Ymake.name])
            if not os.path.exists(ymake_path):
                raise Exception("No ymake file found in path %s" % ymake_path)

            target = os.path.join(self.arcadia, "local.ymake")
            os.symlink(ymake_path, target)
        os.symlink(self.arcadia, self.abs_path("arcadia"))
        if not self.ctx[DebugMode.name]:
            self.update_ya()
            packages_description = self.run_packer()
            with open(os.path.join(self.log_path(), "packages.json"), "w") as fp:
                json.dump(packages_description, fp, indent=4)

        post_deploy_cmd = self.ctx[PostDeployScript.name]
        run_post_deploy_cmd = self.ctx[RunPostDeployScript.name]
        if post_deploy_cmd and run_post_deploy_cmd:
            self.run_post_deploy_script(os.path.join(self.arcadia, post_deploy_cmd))
        else:
            logging.info("No post-deploy script specified")


__Task__ = DeployKiKiMR
