# -*- coding: utf-8 -*-

import logging
import json

import sandbox.common.types.task as ctt
from sandbox import common

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import DictRepeater
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.decorators import retries
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny.client import NannyApiException
from sandbox.projects.common import utils

RETRIES = 10
DELAY = 30

TTL = 12 * 60 * 60  # 12 hours

ALEMATE_RUNNING_GROUP = ('NEW', 'COMMITTED', 'MERGED')
ALEMATE_FAILED_GROUP = ('REJECTED', 'FAILED')
ALEMATE_SUCCESS_GROUP = ('DONE',)
NANNY_UI_URL = "https://nanny.yandex-team.ru/ui/#"


class WaitDeployment(SandboxIntegerParameter):
    name = 'wait_deployment'
    description = 'Период опроса состояния автодеплоя (секунды)'
    default_value = 120


class GetServicesFromRecipe(SandboxBoolParameter):
    name = 'services_from_recipe'
    description = 'Get list of services from recipe'
    default_value = False


class SkipServicesWithoutRelease(SandboxBoolParameter):
    name = 'services_skip_without_release_ticket'
    description = 'Skip service to deploy without release ticket'
    default_value = False


class ReleaseTask(SandboxStringParameter):
    name = 'deployment_task_id'
    description = 'Sandbox task with release resources'
    required = True


class NannyDashboardName(SandboxStringParameter):
    name = 'deployment_nanny_dashboard_name'
    description = 'Название дашборда в nanny для автодеплоя'
    required = True


class NannyDashboardRecipeName(SandboxStringParameter):
    name = 'deployment_nanny_dashboard_recipe'
    description = 'Название метарецепта'
    default_value = 'db_switch'
    required = True


class NannyDashboardFilter(SandboxStringParameter):
    name = 'deployment_nanny_dashboard_filter'
    description = 'Фильтр сервисов в дашборде для автодеплоя'
    required = False


class NannyDashboardItsTask(DictRepeater, SandboxStringParameter):
    name = 'deployment_nanny_dashboard_its_task'
    description = 'Словарь новых значений для ITS ручек'
    required = False


class NannyDashboardSetZkTask(DictRepeater, SandboxStringParameter):
    name = 'deployment_nanny_dashboard_set_zk_task'
    description = 'Словарь новых значений для ZK нод'
    required = False


class SandboxReleaseType(SandboxStringParameter):
    name = 'deployment_release_status'
    description = 'Тип релиза нового ресурса'
    default_value = 'unstable'
    choices = [
        ('stable', 'stable'),
        ('prestable', 'prestable'),
        ('testing', 'testing'),
        ('unstable', 'unstable'),
        ('cancelled', 'cancelled'),
    ]
    required = True


class NannyWaitDeployParameter(SandboxBoolParameter):
    """
        Ждать завершения запущенного деплоя
    """
    name = 'deployment_nanny_bool_wait'
    description = 'Ждать завершения запущенного деплоя'
    default_value = False


class VaultName(SandboxStringParameter):
    name = 'vault_name'
    description = 'Sandbox vault c ID для Nanny'
    default_value = 'nanny-oauth-token'


class VaultOwner(SandboxStringParameter):
    name = 'vault_owner'
    description = 'Владелец sandbox vault'
    default_value = 'MEDIA_DEPLOY'


class SemaphoreName(SandboxStringParameter):
    name = 'semaphore_name'
    description = 'Семафор, лимитирующий одновременные выкатки'


class DeployNannyDashboardDev(SandboxTask):
    """
        Деплой нового зарелиженного ресурса в сервисы через дашборды
    """
    type = 'DEPLOY_NANNY_DASHBOARD_DEV'

    execution_space = 512

    cores = 1

    input_parameters = [
        ReleaseTask,
        NannyDashboardName,
        NannyDashboardRecipeName,
        NannyDashboardFilter,
        NannyDashboardItsTask,
        NannyDashboardSetZkTask,
        SandboxReleaseType,
        NannyWaitDeployParameter,
        VaultName,
        VaultOwner,
        WaitDeployment,
        GetServicesFromRecipe,
        SkipServicesWithoutRelease,
        SemaphoreName,
    ]

    def initCtx(self):
        # set ttl
        self.ctx['kill_timeout'] = TTL

    @property
    def footer(self):
        headline = ''
        if 'request' in self.ctx:
            headline = headline + """
                <h4>Release requests: <a href="{nanny_url}/r/{request_id}/">{request_id}</a></h4>
            """.format(nanny_url=NANNY_UI_URL, request_id=self.ctx['request']['id'])
        if 'deploy_task' in self.ctx:
            headline = headline + """
                <h4>Deploy task: <a href="{0}/services/dashboards/catalog/{1}/deployments/catalog/{2}">{1}/</a></h4>
            """.format(NANNY_UI_URL, self.ctx[NannyDashboardName.name], self.ctx['deploy_task'])
        return headline

    def _get_release_tasks(self):
        return sorted([int(i.strip()) for i in str(self.ctx[ReleaseTask.name]).split(",")])

    def _get_release_requests(self, nanny):
        release_tasks = self._get_release_tasks()
        res = {}
        for release_task in release_tasks:
            res[release_task] = self._get_release_request(nanny, release_task)

        return res

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _get_release_request(self, nanny, release_task):
        self.set_info('Find release request {}'.format(release_task))
        req = {
            'limit': 1,
            'query': {
                'sandboxTaskId': str(release_task),
                'sandboxReleaseType': [self.ctx[SandboxReleaseType.name]]
            },
        }
        res = nanny.find_releases(req)
        if len(res) > 0:
            return res[0]
        raise Exception("Couldn't find release request")

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _get_tickets(self, nanny, request_id, service):
        self.set_info(
            'Get related ticket for {0} release request and {1} service'.format(
                request_id, service))
        find = {
            'limit': 1,
            'query': {
                'releaseId': request_id,
                'status': ['IN_QUEUE', 'COMMITTED', 'DEPLOY_SUCCESS'],
                'serviceId': service,
            }
        }
        res = nanny.filter_tickets(find)
        if len(res) > 0:
            return res[0]
        if utils.get_or_default(self.ctx, SkipServicesWithoutRelease):
            return None
        raise Exception("Couldn't get ticket")

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=NannyApiException)
    def _nannyTaskgroupStatus(self, nanny_client, taskgroup):
        return nanny_client.get_taskgroup_status(taskgroup)['status']

    def _commit_ticket(self, nanny, ticket_id):
        self.set_info('Committing {0} ticket'.format(ticket_id))
        payload = {
            'ticketId': ticket_id,
            'spec': {
                'type': 'COMMIT_RELEASE_REQUEST',
                'commitRelease': {}
            }
        }
        url = '{0}/api/tickets/CreateTicketEvent/'.format(nanny._api_url)
        return nanny._post_url(url, json.dumps(payload))

    def _run_dashboard_recipe(
            self,
            nanny,
            dashboard_name,
            recipe_name,
            snapshots,
            set_its_values=None,
            set_zk_values=None):
        self.set_info(
            'Run meta recipe {0} for {1} dashboard'.format(
                recipe_name, dashboard_name))
        return nanny.deploy_dashboard(
            dashboard_name,
            recipe_name,
            service_snapshots=snapshots,
            set_its_values=set_its_values,
            set_zk_values=set_zk_values)

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _get_dashboard(self, nanny_client, dashboard):
        return nanny_client.get_dashboard(dashboard)

    def task_wait(self, task):
        if task["status"] not in list(
                self.Status.Group.FINISH +
                self.Status.Group.BREAK):
            self.wait_tasks(
                [task["id"]],
                list(self.Status.Group.FINISH + self.Status.Group.BREAK),
                wait_all=True
            )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        if self.ctx[SemaphoreName.name]:
            self.semaphores(ctt.Semaphores(
                acquires=[ctt.Semaphores.Acquire(name=self.ctx[SemaphoreName.name])],
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
            ))

    def on_execute(self):
        vault_key = utils.get_or_default(self.ctx, VaultName)
        vault_owner = utils.get_or_default(self.ctx, VaultOwner)
        wait_deployment = utils.get_or_default(self.ctx, WaitDeployment)
        zk_nodes = utils.get_or_default(self.ctx, NannyDashboardSetZkTask)
        its_values = utils.get_or_default(self.ctx, NannyDashboardItsTask)

        token = self.get_vault_data(vault_owner, vault_key)
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=token,
        )

        # release parent task
        api = common.rest.Client()
        for rt in self._get_release_tasks():
            released_task = api.task[rt][:]

            if released_task["status"] not in (self.Status.RELEASED, self.Status.RELEASING):
                self.task_wait(released_task)
                if released_task["status"] != self.Status.RELEASED:
                    self.create_release(
                        released_task["id"], status=self.ctx[
                            SandboxReleaseType.name])

        # get services from dashboard for deployment
        if 'deployment_services' not in self.ctx:
            logging.debug("Getting services from dashboard")
            self.ctx['deployment_services'] = []

            if utils.get_or_default(self.ctx, GetServicesFromRecipe):
                logging.debug("Try to get services from recipe")
                recipe = nanny_client.get_dashboard_recipe(
                    self.ctx[NannyDashboardName.name],
                    self.ctx[NannyDashboardRecipeName.name]
                )
                deployment_services = set()
                for task in recipe['content']['tasks']:
                    if task['data']['name'] == 'set_snapshot_target_state':
                        deployment_services.add(task['data']['params']['service_id'])
                self.ctx['deployment_services'] = list(deployment_services)
            else:
                logging.debug("Try to get services from dashboard group")
                dashboard_filter_list = None
                if self.ctx[NannyDashboardFilter.name]:
                    dashboard_filter_list = [
                        x.strip() for x in self.ctx[
                            NannyDashboardFilter.name].split(',')]

                for dashboard_group in self._get_dashboard(nanny_client,
                                                           self.ctx[NannyDashboardName.name])['groups']:
                    if dashboard_filter_list and dashboard_group[
                            'id'] not in dashboard_filter_list:
                        logging.debug(
                            'Skip dashboard group: {}'.format(
                                dashboard_group['id']))
                        continue
                    self.ctx['deployment_services'] += dashboard_group['services']

        # get related tickets
        if (
            (len(self.ctx['deployment_services']) > 0 and 'snapshots' not in self.ctx) or
            (len(self.ctx['deployment_services']) != len(self.ctx.get('snapshots', dict())))
        ):
            logging.debug('Got services to deploy: {}'.format(
                ",".join(self.ctx['deployment_services'])))
            if not self.ctx.get('snapshots', None):
                self.ctx['snapshots'] = dict()
            # get release request for released task
            if 'requests' not in self.ctx:
                self.ctx['requests'] = self._get_release_requests(nanny_client)

            for release_task, req in self.ctx.get('requests').iteritems():
                # check is release request is open
                if req['status']['status'] == 'OPEN':
                    # commit only for related services
                    for service in self.ctx['deployment_services']:
                        # get related ticket
                        t = self._get_tickets(nanny_client, req['id'], service)
                        if not t:
                            logging.debug('Ticket for {0} service not found. Skip.'.format(service))
                            continue
                        logging.debug(
                            'Get {0} ticket for {1} service'.format(
                                t, service))
                        # check if ticket already committed
                        spec = t['spec']
                        if not spec['serviceDeployment']['snapshotId']:
                            # commit new ticket
                            t = self._commit_ticket(
                                nanny_client, t['id'])
                            logging.debug(
                                'Committed new {0} ticket for {1} service'.format(
                                    t, service))
                            snapshot = t['ticket']['spec'][
                                'serviceDeployment']['snapshotId']
                        else:
                            snapshot = spec['serviceDeployment']['snapshotId']
                        logging.debug(
                            'Assign {0} snapshot for {1} service'.format(
                                snapshot, service))
                        self.ctx['snapshots'][service] = snapshot
                else:
                    self.set_info(
                        'Release request {0} in unappropriate {1} status'.format(
                            req['id'],
                            req['status']['status']))

        if len(self.ctx.get('snapshots', dict())
               ) > 0 and 'deploy_taskgroup' not in self.ctx:
            # run deployment
            deploy = self._run_dashboard_recipe(
                nanny_client,
                self.ctx[NannyDashboardName.name],
                self.ctx[NannyDashboardRecipeName.name],
                self.ctx['snapshots'],
                set_its_values=its_values,
                set_zk_values=zk_nodes)
            self.ctx['deploy_task'] = deploy['_id']
            self.ctx['deploy_taskgroup'] = deploy['taskgroup_id']
        elif len(self.ctx.get('snapshots', dict())) == 0:
            raise Exception("Couldn't get groups from dashboard")

        if 'deploy_taskgroup' in self.ctx and self.ctx[
                NannyWaitDeployParameter.name]:
            deploy_status = self._nannyTaskgroupStatus(
                nanny_client,
                self.ctx['deploy_taskgroup']
            )
            if deploy_status in ALEMATE_RUNNING_GROUP:
                self.wait_time(wait_deployment)
            elif deploy_status in ALEMATE_FAILED_GROUP:
                raise SandboxTaskFailureError(
                    "Taskgroup {} is failed: {}".format(
                        self.ctx['deploy_taskgroup'],
                        deploy_status))
            elif deploy_status not in ALEMATE_SUCCESS_GROUP:
                raise Exception(
                    "Taskgroup {} has unknown state: {}".format(
                        self.ctx['deploy_taskgroup'], deploy_status))


def create_switching_params(dashboard_name, dashboard_recipe, dashboard_filter, group_name='Switching'):

    class _NannyDashboardName(NannyDashboardName):
        default_value = dashboard_name
        group = group_name

    class _NannyDashboardRecipeName(NannyDashboardRecipeName):
        default_value = dashboard_recipe
        group = group_name

    class _NannyDashboardFilter(NannyDashboardFilter):
        default_value = dashboard_filter
        group = group_name

    return _NannyDashboardName, _NannyDashboardRecipeName, _NannyDashboardFilter


__Task__ = DeployNannyDashboardDev
