# -*- coding: utf-8 -*-

import logging
import urllib
import json

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.nanny import nanny

SLEEP = 300


class NannyServiceNameParameter(SandboxStringParameter):

    """
        Названия сервисов в nanny для автодеплоя
    """
    name = 'deployment_nanny_service_name'
    description = 'Название сервиса в nanny для автодеплоя (перечисление через запятую в порядке переключения)'
    default_value = False
    required = True


class SandboxResourceType(SandboxStringParameter):

    """
        Тип ресурса в sandbox для новых релизов
    """
    name = 'deployment_sandbox_resource_type'
    description = 'Тип ресурса в sandbox для новых релизов'
    default_value = False
    required = True


class DeployNannyReleaseRequests(SandboxTask):

    """
        Деплой нового шардмапа в сервис через Nanny
    """
    type = 'DEPLOY_NANNY_RELEASE_REQUESTS'

    input_parameters = [
        NannyServiceNameParameter,
        SandboxResourceType
    ]

    @property
    def footer(self):
        headline = ''
        if 'release_request' in self.ctx:
            for request in self.ctx['release_request']:
                headline = headline + """
                    <h4>Release requests: <a href="https://nanny.yandex-team.ru/ui/#/r/%(REQUEST_ID)s/">%(REQUEST_ID)s</a></h>
                """ % {'REQUEST_ID': request[u'_id']}
        else:
            headline = 'NO DATA'
        return headline

    def _getRequests(self, nanny):
        # poll new release requests
        request_filter = {
            'filter': urllib.urlencode({
                'status': u'OPEN',
                'type': u'SANDBOX_RELEASE',
                'sandbox_release__release_type': u'stable',
                'sandbox_release__resource_types': self.ctx[SandboxResourceType.name].upper(),
            }),
            'limit': 1,
            'order_by': u'+_id'
        }
        url = '{}/v1/requests/'.format(nanny._api_url)
        for request in nanny._get_url(url, params=request_filter)["result"]:
            yield request

    def _ticket_activate(self, nanny, ticket_id):
        url = '{0}/v1/tickets/{1}/timeline/'.format(nanny._api_url, ticket_id)
        payload = {
            'type': 'ACTIVATE_SERVICE_SNAPSHOT',
            'activate_service_snapshot': {
                'activate_recipe': 'activate',
                'prepare_recipe': 'prepare',
                'apply_mode': 'ALL',
            }
        }
        return nanny._post_url(url, json.dumps(payload))

    def _check_service_state(self, nanny, service_name):
        logging.info("Getting status for %s service" % (service_name))
        current_state = nanny.get_service_current_state(service_name)
        try:
            if current_state[u'content'][u'summary'][u'value'] == u'ONLINE':
                logging.info("Service %s has ONLINE status. Ready to activate" % (service_name))
                return True
            else:
                logging.info("Service %s has %s status. Not ready to activate" % (service_name, current_state[u'content'][u'summary'][u'value']))
                return False
        except:
            raise SandboxTaskFailureError("Couldn't get status for service %s in nanny" %
                (service_name))

    def sleep(self, sleep=SLEEP):
        self.wait_time(sleep)

    def on_execute(self):
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token'),
        )
        nanny_services = self.ctx[NannyServiceNameParameter.name].split(',')
        # get release request
        if 'release_request' not in self.ctx:
            self.ctx['release_request'] = []
            for request in self._getRequests(nanny_client):
                self.ctx['release_request'].append(request)
        if len(self.ctx['release_request']) == 0:
            logging.info("Couldn't find any open release requests")
        else:
            for request in self.ctx['release_request']:
                if request[u'status'] == u'OPEN':
                    logging.info("Get open release request: %s" % (request))
                    for service in nanny_services:
                        logging.info("Getting ticket for %s service from %s request" % (service, request[u'_id']))
                        find = {
                            'limit': 1,
                            'query': {
                                'releaseId': request[u'_id'],
                                'serviceId': service,
                            }
                        }
                        t = nanny_client.filter_tickets(find)[0]
                        logging.info("Receive ticket %s for %s service" % (t[u'_id'], service))
                        status = t['status']['status']
                        if status == u'IN_QUEUE':
                            # ready to activate but we need to acquire service state
                            # activate
                            if self._check_service_state(nanny_client, service):
                                logging.info("Ticket is ready to deploy %s" % (t[u'id']))
                                self._ticket_activate(nanny_client, t[u'id'])
                            self.sleep()
                        elif status in (u'PREPARING', u'SWITCHING'):
                            # still switching
                            logging.info("Service %s is currently deploying" % (service))
                            self.sleep()
                        elif status == u'DEPLOY_SUCCESS':
                            logging.info("Service %s successfully deployed" % (service))
                            continue
                        else:
                            # unexpected status, do nothing
                            raise SandboxTaskFailureError("Unexpected status for %s ticket (%s): %s" %
                                (t[u'id'], service, status))
                        raise SandboxTaskFailureError("Couldn't find any related ticket for service %s in %s release request" %
                            (service, request[u'_id']))
                elif request[u'status'] == u'CLOSED':
                    self.ctx['release_request'].remove(request)
            else:
                logging.info("Task successfully completed!")


__Task__ = DeployNannyReleaseRequests
