# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.nanny import nanny


class ShardmapResource(ResourceSelector):

    """
        Ресурс с шардмапом
    """
    name = 'deployment_shardmap_id'
    description = 'Sandbox resource with releasing shardmap'


class NannyServiceNameParameter(SandboxStringParameter):

    """
        Название сервиса в nanny для автодеплоя
    """
    name = 'deployment_nanny_service_name'
    description = 'Название сервиса в nanny для автодеплоя'
    default_value = False


class NannyAutoDeployParameter(SandboxBoolParameter):
    """
        Включить автодеплой новой базы в сервис (nanny)
    """
    name = 'search_bool_autodeploy'
    description = 'Включить автодеплой новой базы в сервис (nanny)'
    default_value = True


class NannyWaitBeforeDeployParameter(SandboxBoolParameter):
    """
        Ждать завершения текущих активных операций в сервисе перед
        запуском автодеплоя
    """
    name = 'search_bool_wait_before'
    description = 'Ждать завершения активных операций перед автодеплоем'
    default_value = False


class NannyWaitDeployParameter(SandboxBoolParameter):
    """
        Ждать завершения запущенного деплоя
    """
    name = 'search_bool_wait'
    description = 'Ждать завершения запущенного деплоя'
    default_value = True


class DeployNannyShardmap(SandboxTask):

    """
        Деплой нового шардмапа в сервис через Nanny
    """
    type = 'DEPLOY_NANNY_SHARDMAP'

    cores = 1

    input_parameters = [
        NannyServiceNameParameter,
        NannyAutoDeployParameter,
        NannyWaitBeforeDeployParameter,
        NannyWaitDeployParameter,
        ShardmapResource,
    ]

    def _wait_all_tasks(self, nanny_client):
        """
            Ждём завершения всех тасков в сервисе няни
        """
        current_state = nanny_client.get_service_current_state(
            self.ctx[NannyServiceNameParameter.name])
        if current_state[u'content'][u'summary'][u'value'] not in (u'ONLINE'):
            # wait until activating would be completed
            logging.debug(
                "Waiting until ACTIVATING state would be completed in nanny service %s" %
                (self.ctx[NannyServiceNameParameter.name]))
            self.wait_time(30)
        return True

    def _wait_current_task(self, nanny_client, current_snapshot):
        """
            Ждём активации определённого снепшола в сервисе няни
        """
        current_state = nanny_client.get_service_current_state(
            self.ctx[NannyServiceNameParameter.name])
        for snapshot in current_state['content']['active_snapshots']:
            if snapshot[u'snapshot_id'] == current_snapshot:
                if snapshot[u'state'] in ('FAILED', 'CANCELED', 'REJECTED'):
                    raise SandboxTaskFailureError(
                        "For some reason nanny task has been failed: %s" %
                        (current_snapshot))
                elif snapshot[u'state'] not in ('ACTIVE'):
                    # wait until activating would be completed
                    logging.debug(
                        "Waiting until ACTIVATING state for snarshot %s would be completed in nanny service %s" %
                        (self.ctx[
                            NannyServiceNameParameter.name],
                            current_snapshot))
                    return False
                else:
                    return True

    def on_execute(self):

        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token'),
        )

        # get timestamp of shardmap that need to be deploy
        if self.ctx[
                NannyWaitBeforeDeployParameter.name] and 'all_tasks_completed' not in self.ctx:
            self.ctx['all_tasks_completed'] = self._wait_all_tasks(nanny_client)

        if 'new_nanny_snapshot' not in self.ctx:
            deployment_resource = channel.sandbox.get_resource(
                self.ctx[ShardmapResource.name])
            deploy_shardmap_task = channel.sandbox.get_task(
                deployment_resource.task_id)
            nanny_task = nanny_client.update_service_sandbox_resources(
                service_id=self.ctx[
                    NannyServiceNameParameter.name],
                task_type=deploy_shardmap_task.type,
                task_id=str(
                    deployment_resource.task_id),
                deploy=self.ctx[NannyAutoDeployParameter.name])
            if self.ctx[NannyAutoDeployParameter.name]:
                self.ctx['new_nanny_snapshot'] = nanny_task[
                    u'set_target_state'][u'snapshot_id']
        # wait until task is completed
        if 'new_nanny_snapshot' in self.ctx and self.ctx[NannyWaitDeployParameter.name]:
            if not self._wait_current_task(
                    nanny_client, self.ctx['new_nanny_snapshot']):
                self.wait_time(60)


__Task__ = DeployNannyShardmap
