import logging
import json
import pprint
import requests

from sandbox.sandboxsdk.sandboxapi import Sandbox
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import TaskSelector, SandboxBoolParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects import resource_types

from sandbox.projects.common.nanny.nanny import NannyClient
from sandbox.projects.common.deploy_jupiter import update_nanny_service


SERVICE_TOPOLOGY = {
    '3day': {
        'tier': 'SaasFrozen3dayTier0',
        'deploy_service': 'saas_refresh_3day_base_frozen_deploy',
        'base_service': 'saas_refresh_3day_base_frozen',
        'nidx_group': 'MAN_WEB_REFRESH_UNITED_3DAY_FROZEN_BASE_NIDX',
        'search_group': 'MAN_WEB_REFRESH_UNITED_3DAY_FROZEN_BASE',
        'gencfg_topology': 'tags/stable-141-r628',
    },
    'samohod': {
        'tier': 'SaasFrozen3dayTier0',
        'deploy_service': 'saas_refresh_3day_base_frozen_deploy',
        'base_service': 'saas_refresh_3day_base_frozen',
        'nidx_group': 'MAN_WEB_REFRESH_UNITED_3DAY_FROZEN_BASE_NIDX',
        'search_group': 'MAN_WEB_REFRESH_UNITED_3DAY_FROZEN_BASE',
        'gencfg_topology': 'tags/stable-153-r388',
    },
    'video_base': {
        'tier': 'SaasFrozenVideoQuickTier0',
        'deploy_service': 'saas_refresh_frozen_video_base_deploy',
        'base_service': 'saas_refresh_frozen_video_base',
        'nidx_group': 'MAN_VIDEO_QUICK_BASE_NEW_R1_NIDX',
        'search_group': 'MAN_VIDEO_QUICK_BASE_NEW_R1',
        'gencfg_topology': 'tags/stable-109-r540',
    },
}


class DetachServiceIndexTaskId(TaskSelector):
    description = 'The Task (of DETACH_SERVICE_INDEX type) that has created index backup'
    task_type = ['DETACH_SERVICE_INDEX']
    name = 'detach_task'


class SavePrevious(SandboxBoolParameter):
    description = 'Save previous shards generation'
    default_value = True
    name = 'save_previous'


class DeployRefreshFrozen(SandboxTask):
    type = 'DEPLOY_REFRESH_FROZEN'
    description = 'Deploy Fresh shards to the dedicated service'
    input_parameters = [DetachServiceIndexTaskId, SavePrevious]

    def initCtx(self):
        self.execution_space = 100  # 100Mb
        self.ctx['notify_via'] = 'email'
        self.ctx['notify_if_failed'] = 'mcden, osol, saas-sandbox'
        self.ctx['notify_if_finished'] = 'saas-sandbox'

    def on_execute(self):
        logging.info(self.ctx)

        service_descr = None

        for keyword in SERVICE_TOPOLOGY:
            if keyword in _load_service_name(self.ctx.get(DetachServiceIndexTaskId.name)):
                service_descr = SERVICE_TOPOLOGY[keyword]
                break

        if not service_descr:
            raise SandboxTaskFailureError('Detach service has not been recognised {}'.format(
                _load_service_name(self.ctx.get(DetachServiceIndexTaskId.name))
            ))

        nanny_client = NannyClient(api_url=NANNY_API_URL, oauth_token=self.get_vault_data('SEPE-PRIEMKA', 'JUBA_NANNY_TOKEN'))

        with self.memoize_stage.build_replicamap:
            self.ctx['state'] = _load_detach_state(self.ctx.get(DetachServiceIndexTaskId.name))

            if self.ctx.get(SavePrevious.name):
                prev_replicamap = self._get_prev_replicamap(nanny_client, service_descr['deploy_service'])
            else:
                prev_replicamap = None

            replicamap = _build_replicamap(
                self.id,
                service_descr['search_group'],
                service_descr['nidx_group'],
                self.ctx['state'],
                _load_tasks_by_shard(self.ctx.get(DetachServiceIndexTaskId.name)),
                service_descr['tier'],
                prev_replicamap,
                service_descr['gencfg_topology'],
            )

            self.ctx['replicamap'] = replicamap

            with open('replicamap.json', 'w') as fp:
                json.dump(replicamap, fp, indent=4)
                fp.flush()

                resource = self.create_resource(
                    resource_path=fp.name,
                    resource_type=resource_types.REPLICAMAP,
                    description='Replicamap for {}, timestamp {}'.format(
                        service_descr['nidx_group'],
                        self.ctx['state']
                    ),
                )
                self.mark_resource_ready(resource)

            self.ctx['replicamap_resource_id'] = resource.id

        conf_id = self._update_deploy_service(nanny_client, service_descr['deploy_service'], self.ctx['state'])
        self._wait_for_conf(conf_id, self.ctx['replicamap'])
        self._update_search_service(nanny_client, service_descr['base_service'], self.ctx['state'])

    def _update_deploy_service(self, nanny_client, nanny_service, state):
        if 'nanny_deploy_snapshot_id' not in self.ctx:
            snapshot_id = update_nanny_service(
                nanny_client,
                nanny_service,
                state,
                sandbox_files=[{
                    'is_dynamic': False,
                    'local_path': 'replicamap.json',
                    'resource_type': 'REPLICAMAP',
                    'task_id': str(self.id),
                    'task_type': self.type,
                }]
            )

            self.ctx['nanny_deploy_snapshot_id'] = snapshot_id
            self.set_info('Nanny deploy snapshot: {}'.format(snapshot_id))
        else:
            snapshot_id = self.ctx['nanny_deploy_snapshot_id']

        conf_id = None
        if 'nanny_conf_id' not in self.ctx:
            for _ in range(5):
                for snapshot in nanny_client.get_service_current_state(nanny_service)['content']['active_snapshots']:
                    if snapshot['snapshot_id'] == snapshot_id:
                        conf_id = snapshot.get('conf_id')
                        break

                if conf_id:
                    self.ctx['nanny_conf_id'] = conf_id
                    self.set_info(
                        'Configuration: <a href="{1}/search?term=C@{0}&page=instances">{0}</a>'.format(conf_id, CLUSTERSTATE_URL),
                        do_escape=False
                    )
                    break
                else:
                    self.wait_time(5)

            if not conf_id:
                raise SandboxTaskFailureError('Cannot obtain configuration from nanny')
        else:
            conf_id = self.ctx['nanny_conf_id']

        return conf_id

    def _update_search_service(self, nanny_client, nanny_service, state):
        with self.memoize_stage.update_search_service:
            snapshot_id = update_nanny_service(
                nanny_client,
                nanny_service,
                state,
                static_files=[{
                    'local_path': 'vars.conf',
                    'content': 'export YT_STATE={}'.format(state),
                    'is_dynamic': False,
                }]
            )

            self.set_info('Nanny snapshot: {}'.format(snapshot_id))

    def _wait_for_conf(self, conf_id, replicamap):
        with self.memoize_stage.first_long_wait:
            self.wait_time(20 * 60)

        if check_deploy_done(conf_id, replicamap):
            return

        self.wait_time(120)

    def _get_prev_replicamap(self, nanny_client, service):
        service_attrs = nanny_client.get_service_runtime_attrs(service)
        resources = service_attrs['content']['resources']

        for service_resource in resources['sandbox_files']:
            if (
                service_resource['local_path'] == 'replicamap.json'
                and service_resource['resource_type'] == 'REPLICAMAP'
            ):
                sb_resource_list = Sandbox().list_resources(
                    resource_type=resource_types.REPLICAMAP,
                    task_id=service_resource['task_id']
                )
                for sb_resource in sb_resource_list:
                    return self.sync_resource(sb_resource.id)


def update_service_tags(instances, additional_tags):
    tags = instances['extended_gencfg_groups']['tags'][:]

    for t in instances['extended_gencfg_groups']['tags']:
        parts = t.split('=')
        if len(parts) == 2:
            if parts[0] in additional_tags.keys():
                tags.remove(t)

    for t, v in additional_tags.iteritems():
        tags.append('{0}={1}'.format(t, v))

    instances['extended_gencfg_groups']['tags'] = tags
    return instances


def _build_replicamap(task_id, base_group, nidx_group, timestamp, tasks_by_shard, tier, prev_replicamap, gencfg_topology):
    nidx_shards = _get_shards(base_group, nidx_group, gencfg_topology)
    shards = {}
    for shard, nidx_instances in nidx_shards.iteritems():
        parts = shard.split('-')  # e.g. SaasFrozen10dayTier0-3276-6552-000000000
        shard_number = '-'.join(parts[1:3])
        shard_id = '-'.join(parts[0:3] + [str(timestamp)])

        shards[shard_id] = {
            'shard': shard_number,
            'instances': [{'instance': instance, 'policy': 'd'} for instance in nidx_instances],
            'rbtorrent': _get_shard_rbtorrent(tasks_by_shard[shard_number]),
        }

    if prev_replicamap:
        # Add previous shards
        prev_shards = {}
        with open(prev_replicamap) as fp:
            prev_rm = json.load(fp)
            for s in prev_rm['shards']:
                if prev_shards and int(s.split('-')[-1]) > int(prev_shards.keys()[0].split('-')[-1]):
                    prev_shards = {}
                if not prev_shards or int(s.split('-')[-1]) == int(prev_shards.keys()[0].split('-')[-1]):
                    prev_shards[s] = prev_rm['shards'][s]

        shards.update(prev_shards)

    return {
        'meta': {
            'sandbox_task_id': task_id,
            'format_version': 2,
        },
        'shards': shards,
        'config': {
            'slot_size_gb': 120,
            'tier': tier,
            'generation': str(timestamp),
        }
    }


def _get_shards(base_group, nidx_group, tag):
    r = requests.get(GENCFG_API + '/' + tag + '/searcherlookup/groups/' + nidx_group + '/instances')
    if not r.ok:
        r.raise_for_status()

    nidx_instances = {}
    for instance in r.json()['instances']:
        nidx_instances.setdefault(instance['hostname'], instance['port'])

    r = requests.get(GENCFG_API + '/' + tag + '/searcherlookup/groups/' + base_group + '/instances')
    if not r.ok:
        r.raise_for_status()

    shards = {}
    for instance in r.json()['instances']:
        shards.setdefault(instance['shard_name'], [])
        shards[instance['shard_name']].append('{}:{}'.format(instance['hostname'], nidx_instances[instance['hostname']]))

    return shards


def _get_shard_rbtorrent(task_id):
    sandbox = Sandbox()
    for resource in sandbox.list_resources(resource_type=resource_types.RTYSERVER_SEARCH_DATABASE, task_id=task_id):
        return resource.skynet_id
    raise RuntimeError('No resource of type RTYSERVER_SEARCH_DATABASE was found in the task #{}'.format(task_id))


def _load_tasks_by_shard(task_id):
    sandbox = Sandbox()
    return sandbox.get_task(task_id).ctx['task_by_shard']


def _load_service_name(task_id):
    sandbox = Sandbox()
    return sandbox.get_task(task_id).ctx['service_name']


def _load_detach_state(task_id):
    sandbox = Sandbox()
    return sandbox.get_task(task_id).timestamp_start


def check_deploy_done(conf, replicamap):
    term = 'C%40{}'.format(conf)

    try:
        response = requests.get('{}/search?term={}&page=instances&json=da'.format(CLUSTERSTATE_URL, term))
        if not response.ok:
            logging.debug('Got not ok while getting "{}", code: {}', response.url, response.status_code)
            return False
        if response.json()['instances']['ahc'] < 5:  # active hosts count
            logging.debug('Got ahc {} < 5'.format(response.json()['instances']['ahc']))
            return False

        response = requests.get('{}/search?term={}&page=shardtool&json=da'.format(CLUSTERSTATE_URL, term))
        if not response.ok:
            logging.debug('Got not ok while getting "{}", code: {}', response.url, response.status_code)
            return False

        shards = response.json()['shardtool']
        logging.debug(pprint.pformat(shards))

    except RuntimeError as e:
        logging.debug('Got runtime error: {}'.format(e))
        return False
    except IOError as e:
        logging.debug('Got io error: {}'.format(e))
        return False
    except KeyError as e:
        logging.debug('Got key error: {}'.format(e))
        return False

    total = len(replicamap['shards'])
    good = sum(good_shard(shard) for shard in shards)

    return good == total


def good_shard(shard):
    hosts = shard.get('value', {}).get('hosts', {})
    for host in hosts.itervalues():
        for instance in host['instances'].itervalues():
            if instance['status'] == 'DONE':
                return True
    logging.debug('Shard %s is not ready', shard.get('key'))
    return False


NANNY_API_URL = 'http://nanny.yandex-team.ru/'

CLUSTERSTATE_URL = 'http://clusterstate.yandex-team.ru'

# TODO: Get actual stable configuration from Gencfg API (or the source nanny service)
GENCFG_API = 'http://api.gencfg.yandex-team.ru'

__Task__ = DeployRefreshFrozen
