__author__ = 'bykanov'

import errno
import logging
import os
import json
import subprocess
import time

from infra import InfraClient

from sandbox import common
from sandbox import sdk2
from sandbox.common.types.task import Status as s_status
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.DeployResourcesToNanny import deploy_attrs

RESOURCES_STR_MAX_LENGTH = 1024 - 1
SANDBOX_URI = "https://sandbox.yandex-team.ru/task/"


class DeployResourcesToNanny(sdk2.Task):
    """ Deploy Sandbox resources with custom attribute to Nanny services """

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Deploy parameters') as deploy_parameters:
            resources_owner = sdk2.parameters.String('Resources owner (for example, SUGGEST)')
            choice_attribute = sdk2.parameters.String('Attribute for choosing sandbox resources (e.g., name)')
            choice_attribute_values = sdk2.parameters.String('Whitespace separated values for choice attribute (e.g., desktop_base touch_base)')
            sandbox_resources = sdk2.parameters.String(
                'Sandbox resources to deploy. '
                'For example, {"geodata5.bin": {"owner": "GEOBASE", "type": "GEODATA5BIN_STABLE", "attrs": {"released": "stable"}}}'
            )
            vault_user = sdk2.parameters.String('Sandbox vault user having nanny token.', default='SUGGEST', required=True)
            nanny_token_name = sdk2.parameters.String('Nanny oauth token name from Sandbox vault', default='robot-rcss-autoadmin-nanny-oauth', required=True)
            nanny_services = sdk2.parameters.String('Whitespace separated nanny services for sequential deploying', default='suggest_target_r1 suggest_target', required=True)
            export_to_timeline = sdk2.parameters.Bool('Export suggest_web events to infra timeline', default=False)  # SUGGEST-1984

        with sdk2.parameters.Group('Test parameters') as test_parameters:
            arcadia_config_paths = sdk2.parameters.String('Watchdog test`s comma separated config paths in Arcadia', default='')
            git_config_dir = sdk2.parameters.String('Watchdog test`s config directory path in Git', default='')
            git_ssh_key_name = sdk2.parameters.String('Git ssh key name from Sandbox vault', default='')
            git_repo_url = sdk2.parameters.String('Url of git repository with config', default='')
            git_branch = sdk2.parameters.String('Branch of git repository with config', default='')
            test_command = sdk2.parameters.String('Command to test backends. Use special words _INST_, _IHOST_, _IPORT_, _LOGS_DIR_ (e.g., curl _INST_/ping >> _LOGS_DIR_/test.log 2>&1)')
            sandbox_resources_for_test = sdk2.parameters.Resource('Sandbox resources for test command (e.g., binary, config and some test data)', multiple=True)
            allowed_persentage = sdk2.parameters.Integer('Persentage of successfully tested instances to success without symbol % (e.g., 50)', required=True, default=50)
            print_errors_count = sdk2.parameters.Integer('Print errors count for watchdog tests', default=10)
            num_service_to_test = sdk2.parameters.Integer('Number of services to test.', required=True, default=1)
            tvm_src = sdk2.parameters.Integer('Source service id for tvm ticket', default=0)
            tvm_dst = sdk2.parameters.Integer('Destination service id for tvm ticket', default=0)
            tvm_secret_name = sdk2.parameters.String('Vault secret name with tvm secret', default='')

            with sdk2.parameters.RadioGroup('Hostname source for nanny instances') as hostname_source:
                hostname_source.values['container_hostname'] = hostname_source.Value(value='container_hostname')
                hostname_source.values['hostname'] = hostname_source.Value(value='hostname', default=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 1024
        client_tags = common.types.client.Tag.Group.LINUX

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    def get_snapshot_state_comment(self, comment):
        # Cut comment if too large and provide sandbox link
        if len(comment) > RESOURCES_STR_MAX_LENGTH:
            link_to_sandbox = "... {}{}".format(SANDBOX_URI, self.id)
            return comment[:RESOURCES_STR_MAX_LENGTH - len(link_to_sandbox)] + link_to_sandbox
        return comment

    def on_execute(self):
        # Download resources for test command
        if self.Parameters.sandbox_resources_for_test:
            for resource_id in self.Parameters.sandbox_resources_for_test:
                self.download_sandbox_resource(resource_id)

        # Get latest sandbox resources ids
        with self.memoize_stage.get_last_sandbox_resources:
            self.Context.last_resources = self.get_last_sandbox_resources()

        oauth_token = sdk2.Vault.data(self.Parameters.vault_user, self.Parameters.nanny_token_name)
        nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=oauth_token)

        infra = False
        infra_sas_deploy_event_id = 0
        infra_vla_man_deploy_event_id = 0
        service_i = 0
        for nanny_service in self.Parameters.nanny_services.strip().split():
            with self.memoize_stage[nanny_service + '_update_resources']:
                service_resources = nanny_client.get_service_resources(nanny_service)
                logging.debug('%s: Current service_resources status %s', nanny_service, json.dumps(service_resources))

                if self.Parameters.export_to_timeline and nanny_service == 'suggest_web':
                    infra = InfraClient(token=oauth_token, base_url="https://infra-api.yandex-team.ru/v1/")
                    infra_service_id = 722  # suggest_web infra service
                    infra_env_id = 1055  # suggest_web production env

                # Find out do we have to update resources in the nanny service
                now = time.localtime(time.time())
                filtered_last_resources = deploy_attrs.filter_by_deploy_attrs(self.Context.last_resources, now)
                new_service_resources, comment = self.gen_service_resources_and_comment(filtered_last_resources, service_resources)

                # Commit updated resources to the nanny service if we have something to update and activate new shapshot
                if comment:
                    snapshot_info = nanny_client.update_service_resources(nanny_service, {
                        'content': service_resources['content'],
                        'comment': comment,
                        'snapshot_id': new_service_resources['snapshot_id'],
                    })
                    logging.debug('%s: snapshot_info %s', nanny_service, json.dumps(snapshot_info))
                    new_snapshot_id = snapshot_info['runtime_attrs']['_id']

                    event_info = nanny_client.create_event(
                        nanny_service,
                        {
                            'type': 'SET_SNAPSHOT_STATE',
                            'content': {
                                'state': 'ACTIVE',
                                'recipe': 'common',
                                'comment': self.get_snapshot_state_comment(comment),
                                'snapshot_id': new_snapshot_id
                            }
                        }
                    )
                    logging.debug('%s event_info %s', nanny_service, json.dumps(event_info))
                    self.set_info('{}: try to deploy "{}"'.format(nanny_service, comment))

                else:
                    new_snapshot_id = self.get_last_snapshot_id(nanny_client, nanny_service)
                    self.set_info('{}: nothing to deploy, wait for active snapshot.'.format(nanny_service))

                # Wait while nanny service not in ACTIVE state
                while self.get_snapshot_state(nanny_client, nanny_service, new_snapshot_id) != 'ACTIVE':
                    # Set sas deploy event to infra timeline
                    if infra and comment and self.get_snapshot_state(nanny_client, nanny_service, new_snapshot_id) == 'ACTIVATING' and not infra_sas_deploy_event_id:
                        logging.debug('%s stop current infra timeline events', nanny_service)
                        infra.stop_current_service_events(infra_service_id)

                        logging.info('%s set sas deploy event to infra timeline', nanny_service)
                        infra_descr = "{}\nDeploy sandbox task_id: {}\nNanny service: {}".format(comment, self.id, nanny_service)
                        infra_create_resp = infra.create_event(service_id=infra_service_id, env_id=infra_env_id, title=comment, description=infra_descr, datacenters=["sas"])
                        logging.debug('%s created infra deploy event: %s', nanny_service, json.dumps(infra_create_resp))
                        infra_sas_deploy_event_id = infra_create_resp["id"]

                    # Confirm blocked taskgroup job if it requires manual_confirm
                    blocked_job_id = self.find_blocked_taskgroup_job(nanny_client, nanny_service, new_snapshot_id)
                    if blocked_job_id:
                        logging.info('%s: Confirming blocked job %s', nanny_service, blocked_job_id)
                        taskgroup_and_job = blocked_job_id.split('/')
                        nanny_client.send_taskgroup_job_commands(taskgroup_and_job[0], taskgroup_and_job[1], {"type": "confirm"})

                        # Set vla and man deploy event to infra timeline
                        if infra and infra_sas_deploy_event_id and comment and not infra_vla_man_deploy_event_id:
                            logging.debug('%s stop sas deploy infra timeline event with id: %d', nanny_service, infra_sas_deploy_event_id)
                            infra.stop_event(infra_sas_deploy_event_id)

                            logging.info('%s set vla and man deploy event to infra timeline', nanny_service)
                            infra_descr = "{}\nDeploy sandbox task_id: {}\nNanny service: {}".format(comment, self.id, nanny_service)
                            infra_create_resp = infra.create_event(service_id=infra_service_id, env_id=infra_env_id, title=comment, description=infra_descr, datacenters=["vla", "man"])
                            logging.debug('%s created infra deploy event: %s', nanny_service, json.dumps(infra_create_resp))
                            infra_vla_man_deploy_event_id = infra_create_resp["id"]

                    time.sleep(10)

                # Stop deploy infra timeline event
                if infra:
                    if infra_vla_man_deploy_event_id:
                        logging.debug('%s stop vla and man deploy infra timeline event with id: %d', nanny_service, infra_vla_man_deploy_event_id)
                        infra.stop_event(infra_vla_man_deploy_event_id)
                    elif infra_sas_deploy_event_id:
                        logging.debug('%s stop sas deploy infra timeline event with id: %d', nanny_service, infra_sas_deploy_event_id)
                        infra.stop_event(infra_sas_deploy_event_id)

            # Run watchdog test task
            if ((self.Parameters.arcadia_config_paths or self.Parameters.git_config_dir)
                    and service_i < self.Parameters.num_service_to_test):
                with self.memoize_stage[nanny_service + '_watchdog_test']:
                    self.set_info('{}: Run watchdog test task for nanny service instances.'.format(nanny_service))

                    subtask_class = sdk2.Task["SUGGEST_WATCHDOG"]
                    subtask = subtask_class(
                        self,
                        description='Watchdog test for service {}'.format(nanny_service),
                        arcadia_config_paths=self.Parameters.arcadia_config_paths,
                        allowed_persentage=self.Parameters.allowed_persentage,
                        print_errors_count=self.Parameters.print_errors_count,
                        nanny_token_name=self.Parameters.nanny_token_name,
                        vault_user=self.Parameters.vault_user,
                        nanny_service=nanny_service,
                        git_ssh_key_name=self.Parameters.git_ssh_key_name,
                        git_repo_url=self.Parameters.git_repo_url,
                        git_branch=self.Parameters.git_branch,
                        git_config_dir=self.Parameters.git_config_dir,
                        tvm_src=self.Parameters.tvm_src,
                        tvm_dst=self.Parameters.tvm_dst,
                        tvm_secret_name=self.Parameters.tvm_secret_name,
                        hostname_source=self.Parameters.hostname_source,
                        create_sub_task=False
                    ).enqueue()
                    self.Context.watchdog_task_id = subtask.id
                    raise sdk2.WaitTask(subtask, s_status.Group.FINISH | s_status.Group.BREAK, wait_all=True)

                watchdog_task = sdk2.Task[self.Context.watchdog_task_id]
                if watchdog_task.status != s_status.SUCCESS:
                    if 'some_errors' in watchdog_task.Context and watchdog_task.Context.some_errors:
                        for fail_msg in watchdog_task.Context.some_errors:
                            self.set_info(nanny_service + ': ' + fail_msg)
                    raise SandboxTaskFailureError('{}: Watchdog test task is not successful'.format(nanny_service))

                logging.info('Watchdog test task is finished')

            # Run test command
            if self.Parameters.test_command and service_i < self.Parameters.num_service_to_test:
                with self.memoize_stage[nanny_service + '_test_command']:
                    self.set_info('{}: Run test command for service instances.'.format(nanny_service))
                    service_instances = nanny_client.get_service_current_instances(nanny_service)['result']
                    logging.debug('%s: service_instances %s', nanny_service, json.dumps(service_instances))
                    successful_instances_count = 0
                    for service_instance in service_instances:
                        test_command_real = self.replace_instance_vars(str(self.Parameters.test_command), service_instance)
                        logging.debug('%s: Run test command %s', nanny_service, test_command_real)
                        try:
                            test_command_output = subprocess.check_output(test_command_real, stderr=subprocess.STDOUT, shell=True)
                            logging.debug('%s: output of test command "%s" to instance %s: %s', nanny_service, test_command_real, service_instance, test_command_output.decode('utf-8'))
                            successful_instances_count += 1
                        except subprocess.CalledProcessError as e:
                            logging.debug('%s: test command "%s" to instance %s is failed with error %s, return code %s and output %s',
                                          nanny_service, e.cmd, service_instance, repr(e), e.returncode, e.output)
                        except Exception as e:
                            logging.debug('%s: test command "%s" to instance %s is failed with error %s', nanny_service, test_command_real, service_instance, repr(e))

                    successful_percentage = successful_instances_count * 100 / len(service_instances)
                    allowed_percentage = self.Parameters.allowed_persentage
                    if successful_percentage < allowed_percentage:
                        raise SandboxTaskFailureError('{}: Percentage of successful instances {}% less then allowed percentage {}%'.format(nanny_service, successful_percentage, allowed_percentage))

            self.set_info('{}: OK.'.format(nanny_service))

            service_i += 1

    def gen_service_resources_and_comment(self, last_resources, service_resources):
        resources_to_add = last_resources.copy()  # Dict with absent resources in the service for now. We have to add them (not modify).
        resources_to_modify = {}  # Dict with resources to modify. Needed for comment generating
        for s_res_i, s_res_v in enumerate(service_resources['content']['sandbox_files']):
            logging.info('service resource: %s %s ', s_res_i, json.dumps(s_res_v))

            for k, v in last_resources.iteritems():
                if s_res_v['local_path'] == k:
                    logging.info('Found local_path="%s" %s ', s_res_v['local_path'], json.dumps(s_res_v))
                    del resources_to_add[k]

                    # Check, do we have to modify current service resource
                    new_res = self.convert_sandbox_resource_to_service_resource(v, k)
                    if s_res_v['task_id'] != new_res['task_id'] or s_res_v['resource_type'] != new_res['resource_type'] or s_res_v['task_type'] != new_res['task_type']:
                        logging.info('Modify resource values with local_path="%s" to %s ', s_res_v['local_path'], json.dumps(new_res))
                        service_resources['content']['sandbox_files'][s_res_i].update(new_res)
                        resources_to_modify[k] = v.copy()

        # Add new resources
        logging.info('resources_to_add %s ', json.dumps(resources_to_add))
        for k, v in resources_to_add.iteritems():
            new_res = self.convert_sandbox_resource_to_service_resource(v, k)
            service_resources['content']['sandbox_files'].append(new_res)
        logging.info('service_resources just before deploy %s ', json.dumps(service_resources))

        # Create comment
        comment = []
        if resources_to_modify:
            formatted_resources_to_modify = ['{name}({id})'.format(name=k, id=v['id']) for k, v in resources_to_modify.iteritems()]
            comment.append('Change resources: ' + ' '.join(formatted_resources_to_modify))
        if resources_to_add:
            formatted_resources_to_add = ['{name}({id})'.format(name=k, id=v['id']) for k, v in resources_to_add.iteritems()]
            comment.append('Add resources: ' + ' '.join(formatted_resources_to_add))
        comment_ret = '. '.join(comment)
        logging.info('Generated comment %s ', comment_ret)

        return service_resources, comment_ret

    def find_blocked_taskgroup_job(self, nanny_client, nanny_service, snapshot_id):
        active_snapshots = nanny_client.get_service_current_state(nanny_service)['content']['active_snapshots']
        if not active_snapshots:
            return False

        snapshot_info = next((si for si in active_snapshots if si['snapshot_id'] == snapshot_id), None)
        if not snapshot_info:
            return False

        current_taskgroup_id = snapshot_info['taskgroup_id']
        taskgroup_children = nanny_client.get_taskgroup_children(current_taskgroup_id)
        logging.debug('current_taskgroup_id %s taskgroup_children %s ', current_taskgroup_id, json.dumps(taskgroup_children))
        for job in taskgroup_children:
            if job['schedulerOptions']['status'] == 'BLOCKED' and not job['runtimeOptions']['hasConfirm']:
                logging.info('Found blocked job %s', job['id'])
                return job['id']
        return False

    def get_last_snapshot_id(self, nanny_client, nanny_service):
        current_state = nanny_client.get_service_current_state(nanny_service)
        active_snapshots = current_state['content']['active_snapshots']
        if not active_snapshots:
            raise SandboxTaskFailureError('Nothing to change and no snapshots are found in the service.')
        last_snapshot_id = active_snapshots[0]['snapshot_id']
        logging.debug('current_state %s ', json.dumps(current_state))
        return last_snapshot_id

    def get_snapshot_state(self, nanny_client, nanny_service, snapshot_id):
        current_state = nanny_client.get_service_current_state(nanny_service)
        active_snapshots = current_state['content']['active_snapshots']
        if not active_snapshots:
            ret_state = 'NO_SNAPSHOTS'
        snapshot_info = next((si for si in active_snapshots if si['snapshot_id'] == snapshot_id), None)
        if not snapshot_info:
            ret_state = 'NOT_FOUND'
        else:
            ret_state = snapshot_info['state']
        logging.debug('State of snapshot %s is %s', snapshot_id, ret_state)
        return ret_state

    def convert_sandbox_resource_to_service_resource(self, sandbox_resource, local_path):
        logging.debug('try to update local_path=%s with attributes: %s', local_path, json.dumps(sandbox_resource))
        service_resource = {
            'task_id': str(sandbox_resource['task']['id']),
            'resource_type': sandbox_resource['type'],
            'task_type': self.get_task_type_by_task_id(sandbox_resource['task']['id']),
            'local_path': local_path,
            'resource_id': str(sandbox_resource['id'])
        }
        return service_resource

    def get_last_sandbox_resources(self):
        last_resources = {}
        sandbox_client = common.rest.Client()
        if self.Parameters.choice_attribute and self.Parameters.choice_attribute_values:
            for name in self.Parameters.choice_attribute_values.split():
                # Create dict for request
                request = {}
                request['state'] = 'READY'
                if self.Parameters.resources_owner:
                    request['owner'] = self.Parameters.resources_owner
                request['attrs'] = {self.Parameters.choice_attribute: name, 'autodeploy': 'yes'}

                last_res_list = sandbox_client.resource[request, :1]
                if not last_res_list or not last_res_list['items']:
                    logging.info('Could not find resource with attribute "%s: %s"', self.Parameters.choice_attribute, name)
                    continue
                last_res = last_res_list['items'][0]
                logging.info('Found resource: %s ', json.dumps(last_res))
                last_resources[name] = last_res.copy()

        if self.Parameters.sandbox_resources:
            resources = json.loads(self.Parameters.sandbox_resources)
            for name, v in resources.iteritems():
                # Create dict for request
                request = v
                request['state'] = 'READY'

                last_res_list = sandbox_client.resource[request, :1]
                if not last_res_list or not last_res_list['items']:
                    logging.info('Could not find resource %s by filter %s', name, json.dumps(v))
                    continue
                last_res = last_res_list['items'][0]
                logging.info('Found resource: %s ', json.dumps(last_res))
                last_resources[name] = last_res.copy()

        if (not self.Parameters.choice_attribute or not self.Parameters.choice_attribute_values) and not self.Parameters.sandbox_resources:
            raise SandboxTaskFailureError('Please set some filters for resources to deploy')

        return last_resources

    def get_task_type_by_task_id(self, task_id):
        sandbox_client = common.rest.Client()
        task_info = sandbox_client.task[{"id": task_id}, : 1]
        if task_info:
            logging.info('Got task info for task(%s): %s', task_id, json.dumps(task_info))
        task_type = task_info["items"][0]["type"]
        return task_type

    def replace_instance_vars(self, text, service_instance):
        instance_vars = {
            '_INST_': str(service_instance['hostname']) + ':' + str(service_instance['port']),
            '_IHOST_': str(service_instance['hostname']),
            '_IPORT_': str(service_instance['port']),
            '_LOGS_DIR_': str(self.log_path())
        }
        for var in instance_vars:
            text = text.replace(var, instance_vars[var])
        return text

    def download_sandbox_resource(self, resource_id, local_path=''):
        # Download SANDBOX resource and make symlinks in current working directory
        resource_path = str(sdk2.ResourceData(resource_id).path)
        if not local_path:
            local_path = os.path.basename(resource_path)
        logging.info('Downloading resource into %s. Making symlink in current directory as %s', resource_path, local_path)
        self.force_symlink(resource_path, local_path)

    def force_symlink(self, src, dest):
        try:
            os.symlink(src, dest)
        except OSError, e:
            if e.errno == errno.EEXIST:
                os.remove(dest)
                os.symlink(src, dest)
