# coding=utf-8

import json
import logging
import requests
import urlparse
import time


class InfraClient(object):

    def __init__(self, token, base_url="https://infra-api-test.yandex-team.ru/v1/"):
        self.base_url = base_url

        self._session = requests.Session()
        self._session.headers = {"Content-Type": "application/json", "Authorization": "OAuth " + token}

    def create_event(self, service_id, env_id, title, description, ticket_id="", datacenters=None,
                     type_event="maintenance", severity="minor"):
        data = {
            "title": title,
            "description": description,
            "environmentId": env_id,
            "serviceId": service_id,
            "type": type_event,
            "severity": severity,
            "startTime": int(time.time()),
        }
        if datacenters:
            for dc in datacenters:
                data[dc] = True

        return self.request("POST", 'events', data)

    def stop_event(self, event_id):
        assert event_id
        data = {
            "finishTime": int(time.time())
        }
        url_path = "events/{}".format(event_id)
        return self.request("PUT", url_path, data)

    def get_current_service_events(self, service_id):
        assert service_id
        url_path = "events/current?serviceId={}".format(service_id)
        return self.request("GET", url_path)

    def stop_current_service_events(self, service_id):
        assert service_id
        events = self.get_current_service_events(service_id)
        if events:
            logging.debug("Found unclosed infra events for serviceId {}: {}".format(service_id, events))

            for event in events:
                logging.debug("Close infra event with id {}".format(event["id"]))
                self.stop_event(event["id"])
        else:
            logging.debug("Not found unclosed infra events for serviceId {}".format(service_id))

    def request(self, method, url_path, data=None):
        url = urlparse.urljoin(self.base_url, url_path)
        logging.debug("Request to url {}".format(url))

        if data:
            data = json.dumps(data)
            logging.debug("Request data {}".format(data))

        response = self._session.request(method=method, url=url, data=data)
        if response.status_code != 200:
            logging.error("Bad response code {} for {}: {}".format(response.status_code, url, response.text))
            return None

        return response.json()
