# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskErr

from sandbox.projects.DeployUkrop import z2
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build.parameters import ArcadiaUrl

BUILD_TASK_TYPE = 'BUILD_UKROP'


class PkgFullNames(parameters.SandboxParameter):
    name = 'pkg_full_names'
    description = 'List of full package names from build task'


class BuildTaskId(parameters.SandboxIntegerParameter):
    name = 'build_task_id'
    description = '{} task id'.format(BUILD_TASK_TYPE)


class Z2CfgParam(parameters.SandboxStringParameter):
    name = 'z2_config'
    description = 'Z2 config name'
    required = True
    choices = tuple(
            (z2_cfg_id, z2_cfg_id) for z2_cfg_id in z2.get_cfg_props().keys()
    )
    default_value = choices[0][1]


class UseExistingPkgs(parameters.SandboxBoolParameter):
    name = 'use_existing_pkgs'
    description = 'Get packages from existing {} task'.format(BUILD_TASK_TYPE)
    default_value = True
    required = True
    sub_fields = {
        'true': [BuildTaskId.name],
        'false': [ArcadiaUrl.name],
    }


def get_deploy_ukrop_params():
    return [
            UseExistingPkgs,
            ArcadiaUrl,
            BuildTaskId,
            Z2CfgParam,
    ]


class DeployUkrop(task.SandboxTask):

    type = 'DEPLOY_UKROP'

    input_parameters = get_deploy_ukrop_params()

    z2_api_key_vault_name_tpl = '{}-z2-api'

    def on_execute(self):
        if self.ctx[UseExistingPkgs.name]:
            self._deploy_task(self.ctx[BuildTaskId.name])
        elif self.ctx.get('subtask_id'):
            self._deploy_task(self.ctx.pop('subtask_id'))
        else:
            self._create_build_subtask()

    def _create_build_subtask(self):
        build_task_ctx = {
                'build_bundle': False,
                'build_system': consts.YMAKE_BUILD_SYSTEM,
                'clear_build': False,
                'create_packages': True,
                ArcadiaUrl.name: self.ctx[ArcadiaUrl.name],
                'notify_via': '',
                'notify_if_failed': '',
                'notify_if_finished': '',
        }
        logging.info('Create {} sub-task with ctx: {}.'.
                     format(BUILD_TASK_TYPE, build_task_ctx))
        build_task = self.create_subtask(
                task_type=BUILD_TASK_TYPE,
                input_parameters=build_task_ctx,
                description='Child of {} task #{}'.format(self.type, self.id)
        )
        self.ctx['subtask_id'] = build_task.id
        self.wait_task_completed(self.ctx['subtask_id'])

    def _deploy_task(self, build_task_id):
        build_task = self._check_task(build_task_id)
        logging.info('Deploy {} task #{}.'.
                     format(build_task.type, build_task.id))
        config_id = self.ctx[Z2CfgParam.name]
        z2api_key = self._get_z2_api_key(config_id)
        z2api = z2.Z2Api(config_id, z2api_key)
        deploy_props = self._get_deploy_props(build_task)
        logging.info('Deploy props: {}.'.format(deploy_props))
        z2api.edit_items(deploy_props)
        self.set_info('{} edited.'.format(config_id))
        z2api.update()
        self.set_info('{} updated.'.format(config_id))

    def _check_task(self, build_task_id):
        build_task = channel.channel.sandbox.get_task(build_task_id)
        if not build_task:
            raise TaskErr('Cannot get {} task #{}.'.
                          format(BUILD_TASK_TYPE, build_task_id))
        if build_task.type != BUILD_TASK_TYPE:
            raise TaskErr('Task #{}, wrong type: {}.'.
                          format(build_task.id, build_task.type))
        if not build_task.is_finished():
            raise TaskErr('Task #{} is not ready.'.format(build_task.id))
        for ctx_key in [ArcadiaUrl.name, PkgFullNames.name]:
            if ctx_key not in build_task.ctx:
                raise TaskErr('No ctx key {} in task #{}.'.
                              format(ctx_key, build_task.id))
        return build_task

    def _get_deploy_props(self, build_task):
        pkg_names = set()
        pkg_versions = set()
        for pkg_full_name in build_task.ctx[PkgFullNames.name]:
            pkg_name, pkg_version = pkg_full_name.split('=', 1)
            pkg_names.add(pkg_name)
            pkg_versions.add(pkg_version)
        if len(pkg_versions) != 1:
            raise TaskErr('Malformed {} in build task: {}.'.
                          format(PkgFullNames.name,
                                 build_task.ctx[PkgFullNames.name]))
        parsed_url = svn.Arcadia.parse_url(build_task.ctx[ArcadiaUrl.name])
        if parsed_url.trunk:
            branch = 'trunk'
        else:
            branch = '/'.join(['branches', parsed_url.branch])
        return {
                'pkg_names': pkg_names,
                'pkg_version': pkg_versions.pop(),
                'branch': branch,
                'revision': parsed_url.revision,
        }

    def _get_z2_api_key(self, config_id):
        vault_name = self.z2_api_key_vault_name_tpl.format(config_id)
        return self.get_vault_data(self.owner, vault_name)


__Task__ = DeployUkrop
