import json
import logging
import pprint
import urllib
import urllib2

from sandbox.sandboxsdk.errors import SandboxTaskFailureError as TaskErr

from sandbox.projects.DeployUkrop import z2_cfg


def get_cfg_props():
    return z2_cfg.CFG_PROPS


class HTTP(object):
    GET = 'GET'
    POST = 'POST'


class Z2Api(object):

    url_root = 'https://z2.yandex-team.ru/api/v1'
    timeout = 15    # sec

    class Method(object):
        GET_ITEMS = 'items'
        EDIT_ITEMS = 'editItems'
        UPDATE = 'update'

    def __init__(self, config_id, api_key):
        self.__config_id = config_id
        self.__api_key = api_key
        if self.__config_id not in z2_cfg.CFG_PROPS:
            raise TaskErr('Unknown z2 config id: {}.'.format(self.__config_id))
        self.__props = z2_cfg.CFG_PROPS[self.__config_id]

    def _get_auth_params(self):
        return {
                'configId': self.__config_id,
                'apiKey': self.__api_key,
        }

    def get_items(self):
        args = urllib.urlencode(self._get_auth_params())
        subpath = '?'.join([Z2Api.Method.GET_ITEMS, args])
        return self.request(subpath)

    def edit_items(self, deploy_props):
        items = self._mk_items(deploy_props)
        if not items:
            raise TaskErr('No items to update {}.'.format(self.__config_id))
        data = self._get_auth_params()
        data['items'] = json.dumps(items)
        return self.request(Z2Api.Method.EDIT_ITEMS, data)

    def update(self):
        data = self._get_auth_params()
        return self.request(Z2Api.Method.UPDATE, data)

    def request(self, subpath, data=None):
        method = HTTP.POST if data else HTTP.GET
        url = '{root}/{subpath}'.format(root=Z2Api.url_root, subpath=subpath)
        obscured_data = pprint.pformat(data).replace(self.__api_key, '<hidden>')
        logging.info('Z2 API {method} {url} (data: {data}).'.
                     format(method=method, url=url, data=obscured_data))
        enc_data = urllib.urlencode(data) if data else data
        try:
            http_fd = urllib2.urlopen(url, timeout=Z2Api.timeout, data=enc_data)
            json_resp = json.load(http_fd)
        except urllib2.HTTPError as http_err:
            raise TaskErr(http_err.read())
        except Exception as exc:
            exc_msg = 'Failed to {} {}:\n{}.'.format(method, url, exc)
            logging.exception(exc_msg)
            raise TaskErr(exc_msg)
        logging.info('Z2 API raw answer: {}'.format(pprint.pformat(json_resp)))
        if not json_resp.get('success'):
            logging.error('Z2 API error for URL {}'.format(url))
            logging.error('Error message: {}'.format(json_resp.get('errorMsg')))
            logging.error('Stack trace: {}'.format(json_resp.get('stackTrace')))
            raise TaskErr(json_resp)
        return json_resp

    def _mk_items(self, deploy_props):
        z2_items = self.get_items().get('response', {}).get('items')
        if not z2_items:
            raise TaskErr('No items in {}.'.format(self.__config_id))
        z2_names = set([item['name'] for item in z2_items])

        allowed_svn_dirs = self._get_svn_dirs(z2_names)
        logging.info('Allowed svn dirs: {}.'.format(allowed_svn_dirs))

        sandbox_pkg_names = deploy_props['pkg_names']
        z2_sandbox_pkg_names = z2_names & sandbox_pkg_names
        if self._is_all_possible_pkgs_enabled(self.__props['items']['pkg']):
            self.__props['items']['pkg'] = ['{}={{pkg_version}}'.format(name)
                                            for name in z2_sandbox_pkg_names]
        cfg_pkg_names = set([item.split('=')[0]
                             for item in self.__props['items']['pkg']])
        allowed_pkg_names = z2_sandbox_pkg_names & cfg_pkg_names
        logging.info('Allowed pkg names: {}.'.format(allowed_pkg_names))

        items = []
        for raw_item in self.__props['items']['svn'] + \
                self.__props['items']['pkg']:
            raw_name, raw_version = raw_item.split('=', 1)
            svn_dir = raw_name.split(z2_cfg.SVN_DELIM, 1)[0]
            if raw_name in allowed_pkg_names or svn_dir in allowed_svn_dirs:
                name = raw_name.format(**deploy_props)
                version = raw_version.format(**deploy_props)
                items.append({
                    'name': name,
                    'version': version,
                })
        logging.info('New items: {}.'.format(items))
        return items

    def _get_svn_dirs(self, items):
        return set([item.split(z2_cfg.SVN_DELIM, 1)[0]
                    for item in items if z2_cfg.SVN_DELIM in item])

    def _is_all_possible_pkgs_enabled(self, pkgs):
        return bool(len(pkgs) == 1 and pkgs[0] == z2_cfg.ALL_POSSIBLE_PACKAGES)
