import json
import logging
import urllib
import urllib2
import re
import requests
import sys
import time

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sdk2 import yav

logger = logging.getLogger(__name__)


class DataBuilder(object):
    MAX_PACK_SIZE = 50

    def __init__(self):
        self.packsByTs = {}
        self.curPackByTs = {}

    def add_value(self, ts, key, value):
        if ts not in self.packsByTs:
            self.curPackByTs[ts] = []
            self.packsByTs[ts] = [self.curPackByTs[ts]]
        self.curPackByTs[ts].append((key, value))
        if len(self.curPackByTs[ts]) >= self.MAX_PACK_SIZE:
            self.curPackByTs[ts] = []
            self.packsByTs[ts].append(self.curPackByTs[ts])


class SolomonClient(object):

    def __init__(self, oauth_token, project, cluster, service, timeout):
        api_url_format = 'http://solomon.yandex.net/api/v2/push?project={}&cluster={}&service={}'
        self.api_url = api_url_format.format(project, cluster, service)
        self.timeout = timeout
        self.headers = {
            'Authorization': 'OAuth ' + oauth_token,
            'Content-type': 'application/json'
        }

    def send(self, dataPack, ts):
        data = {
            "sensors": []
        }
        for item in dataPack:
            values = item[0].split("|")
            domain = values[0]
            platform = "all"
            if len(values) == 2:
                metric = values[1]
            if len(values) == 3:
                platform = values[1]
                metric = values[2]
            data["sensors"].append({"labels": {"sensor": metric, "domain": domain, "platform": platform},
                                    "ts": float(ts),
                                    "value": float(item[1])})
        try:
            response = requests.post(url=self.api_url, data=json.dumps(data), headers=self.headers, timeout=self.timeout)
            result = response.status_code in [200, 409]
            if not result:
                logger.warn('Bad solomon response status: ' + str(response.status_code))
        except Exception as e:
            logger.warn(str(e))
            result = False
        return result


class GolemStatus:
    OK = "ok"
    WARNING = "warning"
    CRITICAL = "critical"


class GolemEvent:
    def __init__(self, object, name):
        self.object = object
        self.name = name


class GolemClient(object):

    def __init__(self, server, timeout=5.0):
        self.server = server
        self.timeout = timeout

    def updateEvent(self, event, status, description=""):
        params = {
            "object":    event.object,
            "eventtype": event.name,
            "status":    status,
            "info":      description
        }

        failedTries = []
        for t in range(1, 4):
            if t > 0:
                time.sleep(1)  # wait before retry
            try:
                url = urllib2.urlopen(
                    url="http://" + self.server + "/api/events/submit.sbml?" + urllib.urlencode(params),
                    timeout=self.timeout
                )
                response = url.read().strip(" \t\n")
                if response.startswith("ok"):
                    return True
                else:
                    failedTries.append("server response: " + response)

            except urllib2.URLError as urlError:
                try:
                    errResponse = urlError.read()
                    if status == GolemStatus.OK and errResponse.startswith("error: no current event"):
                        return True
                    else:
                        failedTries.append("urllib2.URLError: %s (server response: %s)" % (urlError, errResponse))
                except:
                    failedTries.append("urllib2.URLError: %s" % urlError)

            except urllib2.HTTPError as httpError:
                failedTries.append("urllib2.HTTPError: %s" % httpError)

            except Exception as e:
                failedTries.append("error: %s" % e)

        logger.warn("Failed to update {object: %s, event: %s, status: %s} at %s:" % (event.object, event.name, status, self.server))
        for (t, err) in enumerate(failedTries):
            logger.warn(" [%d] %s" % (t + 1, err))

        return False


class DrawFreshnessPlots(SandboxTask):
    """
        Draw some plots for Freshness
    """

    type = 'DRAW_FRESHNESS_PLOTS'

    cores = 1


    input_parameters = []

    REGIONS = ['ru', 'ua', 'tr', 'by', 'kz', 'com', 'net']

    SURPLUS_WIZARDS = [
        'afisha',
        'afisha_event',
        'anti_mirror',
        'auto_2',
        'autoparts',
        'business_chat_center',
        'buy_tickets',
        'collections-dynamic',
        'companies',
        'default_search',
        'dict_fact',
        'distr',
        'drugs',
        'eda',
        'empty_name',
        'entity-fact',
        'entity_search',
        'ether',
        'fact_instruction',
        'games',
        'geo_common_wizard',
        'graph',
        'health',
        'health_encyclopedia',
        'images',
        'lyrics',
        'maps',
        'market',
        'memorandum',
        'misspell',
        'musicplayer',
        'news',
        'payments',
        'people',
        'poetry_lover',
        'post_indexes',
        'pseudo_fast',
        'rabota',
        'rasp_route',
        'realty',
        'reask',
        'route',
        'special',
        'sport',
        'suggest_fact',
        'time',
        'translate',
        'unianswer',
        'uslugi',
        'video',
        'video_xl',
        'weather',
        'yabs_proxy',
        'yandex_talents',
        'znatoki',
    ]

    SURPLUS_INTENTS = [
        'ALICE',
        'BIATHLON',
        'BUSINESSCHATCENTER',
        'BUYTICKETS',
        'COLLECTIONS_BOARD',
        'DRUGS',
        'EDA',
        'ENTITY_SEARCH',
        'FASTRES',
        'FRESH',
        'GAMES',
        'GEOV',
        'NEWS_WIZARD',
        'ORG_WIZARD_CAROUSEL',
        'PAYMENTS',
        'PEOPLEWIZ',
        'RABOTA',
        'SHINY_DISCOVERY',
        'STATIC_FACT',
        'TRANSPORT',
        'TURBO_SNIPPET',
        'TUTOR',
        'UNITS_CONVERTER',
        'VHS_WIZARD',
        'VIDEOWIZ',
        'WEB_BNA',
        'WEB_NAV',
        'WEB_QUESTION',
        'WEB_ZNATOKI',
        'WIZAFISHA',
        'WIZAFISHAEVENT',
        'WIZAUTO',
        'WIZBROWSERADVFILTER',
        'WIZCOLORS',
        'WIZDISTRICT',
        'WIZDRUGSLANDING',
        'WIZEDADIL',
        'WIZEDAFORBUSINESS',
        'WIZHEALTHENCYCLOPEDIA',
        'WIZIMAGES',
        'WIZLYRICS',
        'WIZMAPS',
        'WIZMARKET',
        'WIZMATH',
        'WIZMUSIC',
        'WIZPOSTINDEXES',
        'WIZRASP',
        'WIZREALTY',
        'WIZREPORTMISC',
        'WIZROUTES',
        'WIZTAXI',
        'WIZTAXIGRUZZ',
        'WIZTAXIWORKCLASSS',
        'WIZTRANSLATE',
        'WIZUSLUGI',
        'WIZWEATHER',
        'WIZWEATHERMONTH',
        'WIZYAAON',
        'WIZYAAON_PHONE_NUMBER',
        'WIZYAFEY',
        'WIZYAFLAG',
        'WIZYAMONEYCARD',
        'WIZYAOFD',
        'WIZYARAYON',
        'WIZYARAYONARENDA',
        'WIZYARAYONNEWS',
        'WIZYAREPETITORQUERY',
        'WIZYARSYA',
        'WIZYATAXI',
        'WIZYATOLOKA_FAST',
        'WIZYATURBO',
        'WIZZNATOKIQUESTION',
        'YABS_PROXY_WIZ_EXP',
        'YA_TALENTS',
        'Y_CONTENTGEN',
        'Y_DISTRICT',
        'Y_EDADEAL',
        'Y_TUTOR',
        'Y_ZNATOKI',
    ]

    SURPLUS_KEYS = ['utility-requests', 'src-Fresh', 'all-wizards', 'all-blended', 'adapter-tv-channel', 'adapter-tv-main', 'src-QUICK_SAMOHOD', 'adapter-news-bno-view']

    CLICKS_KEYS = ['utility-requests', 'src-Fresh', 'wiz-news', 'wiz-video', 'wiz-images', 'src-WebFreshTier', 'src-PlatinumTier0', 'src-WebTier0', 'src-WebTier1', 'src-refresh']
    SURPLUS_SERP_TYPES = ['desktop', 'pad', 'touch']
    CLICKS_SERP_TYPES = ['desktop', 'touch']

    GOLEM_SERVER = 'golem.yandex-team.ru:80'
    GOLEM_OBJECT_NAME = 'freshness-alerts'
    CHECKED_METRICS = {
        'wiz-news': ['ru|touch', 'ru|desktop']
    }
    CHECK_SHOWS_THRESHOLD = 100

    def send_data_to_solomon(self, dataBuilder):
        secret = yav.Secret("sec-01dr8w2ax4x1e5bgnynmk2x1sj") # robot-blendr-priemka_solomon_token
        oauth_token = secret.data()["oauth_token"]
        sender = SolomonClient(
            oauth_token=oauth_token,
            project='freshness',
            cluster='prod',
            service='rt',
            timeout=2
        )
        for ts in sorted(dataBuilder.packsByTs.iterkeys(), reverse=True):
            packs = dataBuilder.packsByTs[ts]
            logger.info('Sending data in solomon for ts=%f (%d packs)' % (ts, len(packs)))
            for pack in packs:
                attempts = 0
                res = False
                while not res and attempts < 3:
                    res = sender.send(pack, ts)
                    attempts += 1

    def get_data_from_rtmr(self, table, keys, num_timestamps=10, max_keys_to_ask=None):
        try:
            result = {}
            max_keys_to_ask = max_keys_to_ask or len(keys)
            while keys:
                reqParams = []
                keys_to_ask = keys[:max_keys_to_ask]
                keys = keys[max_keys_to_ask:]
                for key in keys_to_ask:
                    req = '{"Table":"%s","Key":"%s","MaxRecords":1,"MaxResponseSize":1000000}' % (table, key)
                    reqParams.append('request=' + urllib.quote(req))

                u = urllib2.urlopen('http://rtmr.search.yandex.net:8080/api/v1/tables.json?' + '&'.join(reqParams))
                logger.warn('url length: %s' % len('http://rtmr.search.yandex.net:8080/api/v1/tables.json?' + '&'.join(reqParams)))
                response = u.read()
                if not response:
                    logger.warn('Received empty response from RTMR for key "%s"' % ', '.join(keys_to_ask))
                    continue

                data = json.loads(response)
                if not data:
                    logger.warn('Received bad response from RTMR for key "%s": %s' % (', '.join(keys_to_ask), response))
                    continue

                for chunk in data['Chunks']:
                    metricKey = chunk['Key']
                    metricData = json.loads(chunk['Entries'][0]['Value'])

                    resData = {}
                    lastTimestamps = sorted(metricData.keys(), reverse=True)[:num_timestamps]
                    for ts in lastTimestamps:
                        for subkey in metricData[ts]:
                            # subkey is region or region|device ('ru', 'ru|desktop', etc)
                            if subkey not in resData:
                                resData[subkey] = {}
                            resData[subkey][ts] = metricData[ts][subkey]

                    result[metricKey] = resData

            return result
        except:
            logger.warn('Failed to get key %s from RTMR: %s (%s)' % (key, sys.exc_info()[0], sys.exc_info()[1]))
            return None

    def select_timestamps_to_send(self, metricDataCur, utilityReqsTotal, requestsKey='ReqCnt', minRequests=10000, numTimestamps=3):
        '''
        Return several last timestamps with "good" data.
        We select several timestamps (not only the last one), because there could be quite big lag and data may arrive for several timestamps
        '''
        lastMaxRequests = max(map(lambda r: r[requestsKey], utilityReqsTotal.itervalues()))  # max ReqCnt value during last time (half an hour or hour)
        sortedTs = sorted(metricDataCur.keys(), reverse=True)
        if len(sortedTs) > 1:
            sortedTs.pop(0)
        goodTimestamps = filter(
            lambda ts: (ts in utilityReqsTotal) and (utilityReqsTotal[ts][requestsKey] > minRequests) and (utilityReqsTotal[ts][requestsKey] > lastMaxRequests / 5),
            sortedTs
        )
        return goodTimestamps[:numTimestamps]

    def process_surplus_metrics(self):
        surplus_keys = self.SURPLUS_KEYS + ['wiz-' + w for w in self.SURPLUS_WIZARDS] + ['intent-' + i for i in self.SURPLUS_INTENTS]
        logger.info('process_surplus_metrics: start')
        subkeys = []
        for region in self.REGIONS:
            subkeys.append(region)
            for serpType in self.SURPLUS_SERP_TYPES:
                subkeys.append(region + '|' + serpType)

        data = self.get_data_from_rtmr('fresh_surplus_metrics', surplus_keys, num_timestamps=10, max_keys_to_ask=50)
        utilityReqsData = data.get('utility-requests')
        if not utilityReqsData:
            logger.warn('No utility-requests in table fresh_surplus_metrics')
            return

        dataBuilder = DataBuilder()
        for metricKey in surplus_keys:
            metricData = data.get(metricKey, {})
            for subkey in subkeys:
                if subkey in metricData:
                    for ts in self.select_timestamps_to_send(metricData[subkey], utilityReqsData['']):
                        metricDataForTs = metricData[subkey][ts]
                        if metricKey == 'utility-requests':
                            utilityReqsKey = '%s|%s_%s' % (subkey, metricKey, 'count')
                            dataBuilder.add_value(float(ts), utilityReqsKey, metricDataForTs['ReqCnt'])
                        else:
                            if ts in utilityReqsData.get(subkey, {}):
                                utilityReqsDataForTs = utilityReqsData[subkey][ts]
                                surplus = (metricDataForTs['Win'] - metricDataForTs['Loss']) / utilityReqsDataForTs['ReqCnt']
                                winKey = '%s|%s_%s' % (subkey, metricKey, 'pixel_win')
                                lossKey = '%s|%s_%s' % (subkey, metricKey, 'pixel_loss')
                                surplusKey = '%s|%s_%s' % (subkey, metricKey, 'pixel_win-minus-loss-per-request')
                                countKey = '%s|%s_%s' % (subkey, metricKey, 'pixel_count')
                                dataBuilder.add_value(float(ts), surplusKey, surplus)
                                dataBuilder.add_value(float(ts), winKey, metricDataForTs['Win'])
                                dataBuilder.add_value(float(ts), lossKey, metricDataForTs['Loss'])
                                dataBuilder.add_value(float(ts), countKey, metricDataForTs['Cnt'])
        self.send_data_to_solomon(dataBuilder)
        logger.info('process_surplus_metrics: done')

    def process_clicks_metrics(self):
        logger.info('process_clicks_metrics: start')
        subkeys = []
        for region in self.REGIONS:
            subkeys.append(region)
            for serpType in self.CLICKS_SERP_TYPES:
                subkeys.append(region + '|' + serpType)

        data = self.get_data_from_rtmr('fresh_clicks_metrics', self.CLICKS_KEYS, num_timestamps=30)
        utilityReqsData = data.get('utility-requests')
        if not utilityReqsData:
            logger.warn('No utility-requests in table fresh_clicks_metrics')
            return

        dataBuilder = DataBuilder()
        for metricKey in self.CLICKS_KEYS:
            if metricKey not in data:
                continue
            metricData = data.get(metricKey, {})
            for subkey in subkeys:
                if subkey in metricData:
                    for ts in self.select_timestamps_to_send(metricData[subkey], utilityReqsData['total'], requestsKey='requests'):
                        metricDataForTs = metricData[subkey][ts]
                        if metricKey == 'utility-requests':
                            utilityReqsKey = '%s|%s_%s' % (subkey, metricKey, 'clck_count')
                            dataBuilder.add_value(float(ts), utilityReqsKey, metricDataForTs['requests'])
                        else:
                            if ts in utilityReqsData.get(subkey, {}):
                                utilityReqsLastData = utilityReqsData[subkey][ts]
                                clicksKey = '%s|%s_%s' % (subkey, metricKey, 'clicks')
                                showsKey = '%s|%s_%s' % (subkey, metricKey, 'shows')
                                clicksPerReqKey = '%s|%s_%s' % (subkey, metricKey, 'clicks-per-request')
                                clicksPerReq = (metricDataForTs.get('clicks', 0) / float(utilityReqsLastData['requests'])) if utilityReqsLastData.get('requests') else 0.0

                                ctrKey = '%s|%s_%s' % (subkey, metricKey, 'ctr')
                                ctr = (metricDataForTs.get('clicks', 0) / float(metricDataForTs['shows'])) if metricDataForTs.get('shows') else 0.0
                                dataBuilder.add_value(float(ts), clicksKey, metricDataForTs.get('clicks', 0))
                                dataBuilder.add_value(float(ts), showsKey, metricDataForTs.get('shows', 0))
                                dataBuilder.add_value(float(ts), clicksPerReqKey, clicksPerReq)
                                dataBuilder.add_value(float(ts), ctrKey, ctr)

                                for top in [1, 3, 5, 10]:
                                    hitTopKey = '%s|%s_%s%d' % (subkey, metricKey, 'hit-top', top)
                                    topSubkey = subkey + '|top' + str(top)
                                    if (topSubkey in metricData) and (ts in metricData[topSubkey]):
                                        lastTopData = metricData[topSubkey][ts]
                                        hitTopValue = (lastTopData.get('requests', 0) / float(utilityReqsLastData['requests'])) if utilityReqsLastData.get('requests') else 0.0
                                        dataBuilder.add_value(float(ts), hitTopKey, hitTopValue)

        self.send_data_to_solomon(dataBuilder)
        logger.info('process_clicks_metrics: done')

    def update_metrics_in_golem(self, metricStatuses):
        golemClient = GolemClient(self.GOLEM_SERVER)
        for metric, status, description in metricStatuses:
            eventName = re.sub('\||_', '-', metric) + '-shows'
            event = GolemEvent(self.GOLEM_OBJECT_NAME, eventName)
            golemClient.updateEvent(event, status, description)

    def check_zero_metrics(self):
        logger.info('check_zero_metrics: start')

        data = self.get_data_from_rtmr('fresh_surplus_metrics', self.CHECKED_METRICS.keys(), num_timestamps=6)

        metricStatuses = []
        for metricKey in self.CHECKED_METRICS.iterkeys():
            metricData = data.get(metricKey, {})
            for subkey in self.CHECKED_METRICS[metricKey]:
                metricName = '%s|%s' % (subkey, metricKey)
                if subkey in metricData:
                    totalShows = sum(
                        map(lambda d: d['Cnt'], metricData[subkey].itervalues())
                    )
                    if totalShows > self.CHECK_SHOWS_THRESHOLD:
                        metricStatuses.append((metricName, GolemStatus.OK, ''))
                    else:
                        description = '%s: too little shows (%d) for last 30 minutes' % (metricName, totalShows)
                        metricStatuses.append((metricName, GolemStatus.CRITICAL, description))
                else:
                    description = '%s: no data in RTMR' % metricName
                    metricStatuses.append((metricName, GolemStatus.CRITICAL, description))

        self.update_metrics_in_golem(metricStatuses)
        logger.info('check_zero_metrics: done')

    def on_execute(self):
        self.process_surplus_metrics()
        self.process_clicks_metrics()
        self.check_zero_metrics()


__Task__ = DrawFreshnessPlots
