# -*- coding: utf-8 -*-

import logging
import json

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common.middlesearch import single_host
from sandbox.projects.common import utils
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import apihelpers
from sandbox.projects.common.search import compare_middle_utils as cmu
from sandbox.projects.common.search.eventlog import eventlog
from sandbox.projects.DumpEventlogSingleHost import Requests


shoot_params = cmu.create_shooting_params()
stat_params = cmu.create_eventlog_stat_params()
res_params = cmu.create_resource_params()
eventlog_res_params = cmu.create_eventlog_resource_params()


class DumpAndCalcEventlog(SandboxTask):
    type = 'DUMP_AND_CALC_EVENTLOG'

    input_parameters = (
        single_host.PARAMS +
        (Requests,) +
        shoot_params.params +
        stat_params.params
    )

    def on_enqueue(self):
        self.ctx['out_json_resource_id'] = self.create_resource(
            self.descr, 'stats.json',
            resource_types.EVENTLOG_JSON_STATS
        ).id

    def on_execute(self):
        subtasks = channel.sandbox.list_tasks(parent_id=self.id)
        if not subtasks:
            self._schedule_dump_eventlog_single_host_task()
            self._make_calculations()
            self.wait_all_tasks_completed(channel.sandbox.list_tasks(parent_id=self.id))
        else:
            utils.check_if_tasks_are_ok(subtasks)
            self._get_resource_from_last_calc()

    def _schedule_dump_eventlog_single_host_task(self):
        dst_ctx = {
            'notify_via': '',
            Requests.name: self.ctx[Requests.name]
        }

        for param in single_host.PARAMS:
            dst_ctx[param.name] = self.ctx[param.name]
        utils.copy_param_values(self.ctx, shoot_params, dst_ctx, shoot_params)

        desc = '{}, dump eventlogs for middlesearch with two basesearches'.format(self.descr)
        task = self.create_subtask(
            task_type='DUMP_EVENTLOG_SINGLE_HOST',
            description=desc,
            input_parameters=dst_ctx,
            arch='linux',
        )
        logging.info('task scheduled, id = {}'.format(task.id))
        self.ctx['dump_eventlog_task_id'] = task.id

    def _make_calculations(self):
        evlogs = channel.sandbox.get_task(self.ctx['dump_eventlog_task_id']).ctx[cmu.KEY_OUT_EVENTLOGS]
        for key, res_id in sorted(evlogs.items(), key=lambda (i, j): i):
            run, repeat = [int(x) for x in key.split(',')]
            self.ctx['last_calc_task_id'] = self._schedule_calc_stats_task(run, repeat, res_id)

    def _schedule_calc_stats_task(self, run, repeat, eventlog_res_id):
        logging.info('scheduling CALC_EVENTLOG_STATS task, run # {}, repeat # {}'.format(run, repeat))

        dst_ctx = {
            'notify_via': '',
            eventlog_res_params.Eventlog.name: eventlog_res_id,
            res_params.Evlogdump.name: eventlog.get_evlogdump_resource().id
        }
        utils.copy_param_values(self.ctx, stat_params, dst_ctx, stat_params)

        desc = '{}, calc eventlog stats, run # {}, repeat # {}'.format(
            self.descr, run, repeat
        )
        task = self.create_subtask(
            task_type='CALC_EVENTLOG_STATS',
            description=desc,
            input_parameters=dst_ctx
        )
        logging.info('task scheduled, id = {0}'.format(task.id))
        return task.id

    def _get_resource_from_last_calc(self):
        json_res_id = apihelpers.list_task_resources(
            self.ctx['last_calc_task_id'],
            resource_type="EVENTLOG_JSON_STATS"
        )[0].id
        stats = fu.read_file(self.sync_resource(json_res_id))
        self.ctx["last_calc_stats"] = json.loads(stats)
        out_json_resource = channel.sandbox.get_resource(self.ctx['out_json_resource_id'])
        fu.write_file(out_json_resource.path, stats)


__Task__ = DumpAndCalcEventlog
