# -*- coding: utf-8 -*-


import os
import time
import shutil
import logging
from threading import Thread

from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.projects.common.search.components import Middlesearch
from sandbox.projects.common.search.components import DEFAULT_MIDDLESEARCH_PORT
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.common.dolbilka import convert_queries_to_plan
from sandbox.projects.common.dolbilka import DolbilkaExecutor
from sandbox.projects.common.dolbilka import DolbilkaExecutorMode
from sandbox.projects.common.search import compare_middle_utils as cmu


res_params = cmu.create_resource_params(use_int=True)
res_params.params = tuple([p for p in res_params.params if p != res_params.Evlogdump])
shooting_params = cmu.create_shooting_params()


class RunParallelClone(SandboxBoolParameter):
    name = 'run_parallel_clone'
    description = 'Run parallel middlesearch instance'
    group = 'Other params'
    default_value = False


class DumpEventlog(SandboxTask):
    type = 'DUMP_EVENTLOG'


#    environment = [SandboxDolbilkaEnvironment()]

    input_parameters = res_params.params + shooting_params.params + (RunParallelClone,)

    # list of all used self. params
    #     _archive_model
    #     _eventlog_file
    #     _is_int
    #     _loadlog_file
    #     _middle_binary
    #     _middle_config
    #     _middle_data
    #     _middle_process
    #     _nocache_plan_file
    #     _nrepeats
    #     _nruns
    #     _plan_file
    #     _planner_binary
    #     _port
    #     _pure_dir
    #     _rearrange_dir
    #     _reqs_count
    #     _reqs_file
    #     _rps
    #     _work_dir

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        nruns = self.ctx[shooting_params.Nruns.name]
        nrepeats = self.ctx[shooting_params.Nrepeats.name]

        def create_resources(suffix=None):
            key = cmu.KEY_OUT_EVENTLOGS
            if suffix:
                key += '_' + suffix

            self.ctx[key] = {}
            for irun in xrange(nruns):
                for irepeat in xrange(nrepeats if nruns == 1 + irun else 1):
                    res_descr = '{0}, session {1}, attempt {2}'.format(self.descr, irun, irepeat)
                    res_file = 'eventlog.{0}.{1}'.format(irun, irepeat)
                    if suffix:
                        res_descr += ', ' + suffix
                        res_file += '.' + suffix
                    resource = self._create_resource(res_descr, res_file, resource_types.EVENTLOG_DUMP)
                    self.ctx[key]["{},{}".format(irun, irepeat)] = resource.id
        create_resources()
        if self.ctx[RunParallelClone.name]:
            create_resources('clone')

    def on_execute(self):
        self.do_prepare()
        nruns = self.ctx[shooting_params.Nruns.name]
        for irun in xrange(nruns):
            self.do_run(irun)

    def do_prepare(self):
        self._port = DEFAULT_MIDDLESEARCH_PORT
        if self.ctx[RunParallelClone.name]:
            self._port_clone = self._port + 1
        self._is_int = self.ctx.get(res_params.UseInt.name, False)
        self.create_dirs()
        self.prepare_paths()
        self.make_plans()

    def do_run(self, irun):
        logging.info('run # {0}'.format(irun))

        self.start_middlesearch()

        d_executor = DolbilkaExecutor()
        d_executor.mode = DolbilkaExecutorMode.PLAN_MODE
        d_executor.requests = 500
        dolb_thread = _run_dolbilka_thread(
            d_executor,
            self._nocache_plan_file,
            self._middle_process,
            'warmup',
            run_once=True
        )

        if self.ctx[RunParallelClone.name]:
            d_executor_clone = DolbilkaExecutor()
            d_executor_clone.mode = DolbilkaExecutorMode.PLAN_MODE
            d_executor_clone.requests = 500
            clone_dolb_thread = _run_dolbilka_thread(
                d_executor_clone,
                self._nocache_plan_file,
                self._middle_process_clone,
                'warmup',
                run_once=True
            )
            clone_dolb_thread.join()

        dolb_thread.join()

        self.clear_eventlog()

        nruns = self.ctx[shooting_params.Nruns.name]
        nrepeats = self.ctx[shooting_params.Nrepeats.name] if nruns == 1 + irun else 1
        for irepeat in xrange(nrepeats):
            logging.info('repeat # {0}.{1}'.format(irun, irepeat))

            d_executor.requests = self._reqs_count
            dolb_thread = _run_dolbilka_thread(
                d_executor,
                self._nocache_plan_file if 0 == irepeat else self._plan_file,
                self._middle_process,
                'shot_{}_{}'.format(irun, irepeat),
                run_once=True
            )

            if self.ctx[RunParallelClone.name]:
                d_executor_clone.requests = self._reqs_count
                clone_dolb_thread = _run_dolbilka_thread(
                    d_executor_clone,
                    self._nocache_plan_file if 0 == irepeat else self._plan_file,
                    self._middle_process_clone,
                    'shot_{}_{}'.format(irun, irepeat),
                    run_once=True
                )
                clone_dolb_thread.join()

            dolb_thread.join()

            self.store_resources(irun, irepeat)
            self.clear_eventlog()

        self.stop_middlesearch()

    def create_dirs(self):
        self._work_dir = self.abs_path('workdir')
        if os.path.exists(self._work_dir):
            shutil.rmtree(self._work_dir)
        os.mkdir(self._work_dir)

    def prepare_paths(self):
        self._middle_binary, self._middle_config, self._middle_data,  self._reqs_file, self._archive_model = [
            (self._read_resource(self.ctx[name]).abs_path() if self.ctx[name] else None) for name in [
                res_params.Binary.name, res_params.Config.name,
                res_params.Data.name, res_params.Requests.name, res_params.ArchiveModel.name]]

        if self._middle_data is None:
            self._pure_dir = None
        else:
            self._pure_dir = os.path.join(self._middle_data, 'pure')

        if self._middle_data is None:
            self._rearrange_dir = None
        else:
            self._rearrange_dir = os.path.join(self._middle_data, 'rearrange')

        self._reqs_count, self._rps, self._nruns, self._nrepeats = [
            self.ctx[name] for name in [
                shooting_params.ReqCount.name, shooting_params.Rps.name,
                shooting_params.Nruns.name, shooting_params.Nrepeats.name]]

        self._plan_file = os.path.join(self._work_dir, 'plan.bin')
        self._nocache_plan_file = os.path.join(self._work_dir, 'nocache_plan.bin')

        self._eventlog_file = os.path.join(self._work_dir, 'eventlog.bin')
        self._loadlog_file = os.path.join(self._work_dir, 'load.log')
        if self.ctx[RunParallelClone.name]:
            self._eventlog_file_clone = self._eventlog_file + '.clone'
            self._loadlog_file_clone = self._loadlog_file + '.clone'

    def make_plans(self):
        convert_queries_to_plan(
            self._reqs_file,
            self._plan_file,
            alter_query=lambda q: q + '&nofastcache=1',
            rps=self._rps
        )
        convert_queries_to_plan(
            self._reqs_file,
            self._nocache_plan_file,
            alter_query=lambda q: q + '&nofastcache=1&nocache=da',
            rps=self._rps
        )

    def start_middlesearch(self):
        def impl(port, eventlog_file, loadlog_file):
            middle_process = Middlesearch(
                is_int=self._is_int,
                work_dir=self._work_dir,
                binary=self._middle_binary,
                config_file=self._middle_config,
                pure_dir=self._pure_dir,
                rearrange_dir=self._rearrange_dir,
                archive_model_path=self._archive_model,
                event_log=eventlog_file,
                load_log=loadlog_file,
                port=port,
                patch_queues_size=False,
                query_cache=os.path.join(self._work_dir, 'query_cache_{}'.format(port))
            )
            middle_process.start()
            middle_process.wait()
            return middle_process

        self._middle_process = impl(self._port, self._eventlog_file, self._loadlog_file)
        if self.ctx[RunParallelClone.name]:
            self._middle_process_clone = impl(self._port_clone, self._eventlog_file_clone, self._loadlog_file_clone)

    def stop_middlesearch(self):
        self._middle_process.stop()
        self._middle_process = None
        if self.ctx[RunParallelClone.name]:
            self._middle_process_clone.stop()
            self._middle_process_clone = None

    def clear_eventlog(self):
        with open(self._eventlog_file, 'w') as f:
            f.truncate(0)
        if self.ctx[RunParallelClone.name]:
            with open(self._eventlog_file_clone, 'w') as f:
                f.truncate(0)

    def store_resources(self, irun, irepeat):
        def impl(path, suffix=None):
            key = cmu.KEY_OUT_EVENTLOGS
            log_addition = ''
            if suffix:
                key += '_' + suffix
                log_addition = ', ' + suffix

            logging.info('store eventlog {0}_{1}{2}'.format(irun, irepeat, log_addition))
            eventlog_resource = channel.sandbox.get_resource(self.ctx[key]["{},{}".format(irun, irepeat)])
            logging.info('copying from {0} to {1}{2}'.format(self._eventlog_file, eventlog_resource.path, log_addition))
            shutil.copyfile(path, eventlog_resource.path)
            logging.info('mark resource {0} ready{1}'.format(eventlog_resource.id, log_addition))
            self.mark_resource_ready(eventlog_resource)
        impl(self._eventlog_file)
        if self.ctx[RunParallelClone.name]:
            impl(self._eventlog_file_clone, 'clone')


def _get_time():
    return time.strftime("%d %b %Y %H:%M:%S", time.localtime())


class _DolbilkaThread(Thread):
    def __init__(self, dolbilka, *largs, **dargs):
        Thread.__init__(self)
        self.dolbilka, self.largs, self.dargs = dolbilka, largs, dargs

    def run(self):
        self.dolbilka.run_session_and_dumper(*self.largs, **self.dargs)


def _run_dolbilka_thread(dolbilka, *largs, **dargs):
    thrd = _DolbilkaThread(dolbilka, *largs, **dargs)
    thrd.start()
    return thrd


__Task__ = DumpEventlog
