# -*- coding: utf-8 -*-


import os
import shutil
import logging
from threading import Thread

from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common.middlesearch import single_host
from sandbox.projects.common.dolbilka import convert_queries_to_plan
from sandbox.projects.common.dolbilka import DolbilkaExecutor
from sandbox.projects.common.dolbilka import DolbilkaExecutorMode
from sandbox.projects.common.search import compare_middle_utils as cmu

# TODO: replace basesearches with fake sources

shoot_params = cmu.create_shooting_params()


class Requests(parameters.ResourceSelector):
    name = "middlesearch_request_resource_id"
    description = 'Middlesearch requests:'
    resource_type = 'PLAIN_TEXT_QUERIES'
    group = 'Middlesearch shooting parameters'
    required = True


class DumpEventlogSingleHost(single_host.MiddlesearchSingleHostTask):
    type = 'DUMP_EVENTLOG_SINGLE_HOST'

    input_parameters = (
        single_host.PARAMS +
        (Requests,) +
        shoot_params.params
    )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        channel.task = self
        nruns = self.ctx[shoot_params.Nruns.name]
        key = cmu.KEY_OUT_EVENTLOGS

        self.ctx[key] = {}
        for irun in xrange(nruns):
            for irepeat in xrange(self.ctx[shoot_params.Nrepeats.name] if nruns == 1 + irun else 1):
                res_descr = '{0}, session {1}, attempt {2}'.format(self.descr, irun, irepeat)
                res_file = 'eventlog.{0}.{1}'.format(irun, irepeat)
                resource = self.create_resource(
                    res_descr, res_file, resource_types.EVENTLOG_DUMP
                )
                self.ctx[key]["%d,%d" % (irun, irepeat)] = resource.id

    def _get_middlesearch_additional_params(self):
        self._work_dir = self.abs_path()
        self._prepare_paths()
        self._make_plans()
        return {
            'patch_queues_size': False,
            'event_log': self._eventlog_file
        }

    def _use_middlesearch_component(self, middlesearch):
        nruns = self.ctx[shoot_params.Nruns.name]
        for irun in xrange(nruns):
            self._do_runs(middlesearch, irun, nruns)

    def _do_runs(self, middlesearch, irun, nruns):
        logging.info('middlesearch run = {}'.format(irun))
        middlesearch.start()
        middlesearch.wait()

        d_executor = DolbilkaExecutor()
        logging.info('middlesearch warmup run')
        self._run_dolbilka(d_executor, middlesearch, 500, 'warmup', self._nocache_plan_file)
        self._clear_eventlog()

        nrepeats = self.ctx[shoot_params.Nrepeats.name] if nruns == 1 + irun else 1
        for irepeat in xrange(nrepeats):
            logging.info('repeat # {0}.{1}'.format(irun, irepeat))
            self._run_dolbilka(
                d_executor,
                middlesearch,
                self.ctx.get(shoot_params.ReqCount.name),
                'shot_%d_%d' % (irun, irepeat),
                self._nocache_plan_file if irepeat == 0 else self._plan_file
            )
            self._store_resources(irun, irepeat)
            self._clear_eventlog()

        middlesearch.stop()

    def _run_dolbilka(self, d_executor, middlesearch, requests, name, plan):
        d_executor.mode = DolbilkaExecutorMode.PLAN_MODE
        d_executor.requests = requests
        dolb_thread = _run_dolbilka_thread(
            d_executor,
            plan,
            middlesearch,
            name,
            run_once=True
        )
        dolb_thread.join()

    def _prepare_paths(self):
        self._reqs_file = self._read_resource(self.ctx[Requests.name]).abs_path()

        self._plan_file = os.path.join(self._work_dir, 'plan.bin')
        self._nocache_plan_file = os.path.join(self._work_dir, 'nocache_plan.bin')

        self._eventlog_file = os.path.join(self._work_dir, 'eventlog.bin')

    def _make_plans(self):
        convert_queries_to_plan(
            self._reqs_file,
            self._plan_file,
            alter_query=lambda q: q + '&nofastcache=1',
            rps=self.ctx[shoot_params.Rps.name]
        )
        convert_queries_to_plan(
            self._reqs_file,
            self._nocache_plan_file,
            alter_query=lambda q: q + '&nofastcache=1&nocache=da',
            rps=self.ctx[shoot_params.Rps.name]
        )

    def _clear_eventlog(self):
        with open(self._eventlog_file, 'w') as f:
            f.truncate(0)

    def _store_resources(self, irun, irepeat):
        key = cmu.KEY_OUT_EVENTLOGS

        logging.info('store eventlog {0}_{1}'.format(irun, irepeat))
        eventlog_resource = channel.sandbox.get_resource(self.ctx[key]["%d,%d" % (irun, irepeat)])
        logging.info('copying from {0} to {1}'.format(
            self._eventlog_file, eventlog_resource.path
        ))
        shutil.copyfile(self._eventlog_file, eventlog_resource.path)
        self.mark_resource_ready(eventlog_resource)


class _DolbilkaThread(Thread):
    def __init__(self, dolbilka, *largs, **dargs):
        Thread.__init__(self)
        self.dolbilka, self.largs, self.dargs = dolbilka, largs, dargs

    def run(self):
        self.dolbilka.run_session_and_dumper(*self.largs, **self.dargs)


def _run_dolbilka_thread(dolbilka, *largs, **dargs):
    thrd = _DolbilkaThread(dolbilka, *largs, **dargs)
    thrd.start()
    return thrd


__Task__ = DumpEventlogSingleHost
