# -*- coding: utf-8 -*-
"""Prepare Cachback report and send to StatFace."""

import os
import json
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess as sp

RESULTS_PATH = '//home/edadeal/analytics/reports/cashback_daily'


class CASHBACK_REPORT_TOOL(sdk2.Resource):
    """
        CashbackReportTool binary prepare YT tables and send data to statFace.
    """
    any_arch = True
    executable = True
    releasers = ['antonka', 'adsergeev']


class CashbackReport(sdk2.Task):
    """Prepare Cachback report and send to StatFace."""
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 14400

        start_date = sdk2.parameters.String(
            'Log date to start calculations from.',
            default=''
        )
        end_date = sdk2.parameters.String(
            'Log date to calculate until.',
            default=''
        )
        user_table = sdk2.parameters.String(
            'Banned and blocked users table.',
            default=''
        )
        item_table = sdk2.parameters.String(
            'Bought items table.',
            default=''
        )
        result_table = sdk2.parameters.String(
            'Result output table.',
            default=''
        )

    @staticmethod
    def create_statface_config(creds):
        username = creds['username']
        password = creds['password']
        home = os.path.expanduser('~')
        statbox_dir = home + '/.statbox'
        statbox_config = statbox_dir + '/statface_auth.yaml'

        os.mkdir(statbox_dir, 0755)

        with os.fdopen(os.open(statbox_config, os.O_WRONLY | os.O_CREAT, 0664), 'w') as f:
            f.write('---\n')
            f.write('username: %s\n' % username)
            f.write('password: %s\n' % password)

    def on_execute(self):
        cmd_args = []

        for key, value in (
            ['--start-date', self.Parameters.start_date],
            ['--end-date', self.Parameters.end_date],
            ['--user-table', self.Parameters.user_table],
            ['--item-table', self.Parameters.item_table],
            ['--result-table', self.Parameters.result_table],
        ):
            if value:
                cmd_args.extend([key, value])

        os.environ['YT_TOKEN'] = sdk2.Vault.data('antonka', 'YT_TOKEN')
        os.environ['YQL_TOKEN'] = sdk2.Vault.data('antonka', 'YQL_TOKEN')
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'
        os.environ['robot_wiz_quality'] = sdk2.Vault.data('robot_wiz_quality')
        statface_credentials = json.loads(sdk2.Vault.data('STATFACE_CREDENTIALS'))
        self.create_statface_config(statface_credentials)

        cashback_report_binary_resources = sdk2.Resource.find(
            resource_type=CASHBACK_REPORT_TOOL,
            state='READY'
        ).limit(3)
        last_binary = sorted(list(cashback_report_binary_resources), key=lambda x: x.created)[-1]
        last_binary_path = str(sdk2.ResourceData(last_binary).path)

        logging.info('Using CashbackReportTool binary created on: ' + last_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', last_binary_path, cmd_args, '=======\n\n')

        with sdk2.helpers.ProcessLog(self, logger='CashbackReport') as pl:
            sp.check_call([last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT)
            # p = sp.Popen(process_description, stdout=pl.stdout, stderr=pl.stderr)
            # code = p.wait()

        # if code != 0:
        #     raise RuntimeError("Process exited with code {}. Check OUT and ERR log for logger {}".format(
        #         code,
        #         self.SUBPROCESS_RUNNER_NAME
        #     ))
        # else:
        #     self.upload_to_statface()

        logging.info("Successfully finished process execution")
