# -*- coding: utf-8 -*-
'''Calculate edadeal funnels table and write to YT.'''

import os
import logging

from sandbox import sdk2
import sandbox.common.types.resource as ctr
from sandbox.sdk2.helpers.process import subprocess as sp
from sandbox.projects.Edadeal import Analytics


class EDADEAL_FUNNELS_TOOL(sdk2.Resource):
    '''EdadealFunnelsTool binary to prepare YT funnels.'''

    any_arch = True
    executable = True
    releasers = ['antonka', 'adsergeev']


class EdadealFunnels(sdk2.Task):
    '''Prepare EdadealFunnels table.'''
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 21600

        start_date = sdk2.parameters.String(
            'Range start date to calculate funnels for.',
            default=''
        )
        end_date = sdk2.parameters.String(
            'Range end date to calculate funnels for.',
            default=''
        )
        sessions_folder = sdk2.parameters.String(
            'Edadeal sessions folder.',
            default=''
        )
        funnels_folder = sdk2.parameters.String(
            'Folder to store funnels in.',
            default=''
        )
        create_link = sdk2.parameters.Bool(
            'Create a symlink for the last date.',
            default=True
        )

    def on_execute(self):
        cmd_args = []

        # String parameters
        for key, value in (
            ['--start-date', self.Parameters.start_date],
            ['--end-date', self.Parameters.end_date],
            ['--sessions-folder', self.Parameters.sessions_folder],
            ['--funnels-folder', self.Parameters.funnels_folder],
        ):
            if value:
                cmd_args.extend([key, value])

        # Boolean parameters (flags)
        for key, value in (
            ['--create-link', self.Parameters.create_link],
        ):
            if value:
                cmd_args.append(key)

        os.environ['YT_TOKEN'] = sdk2.Vault.data('ROBOT_EDDL_ANALYST_YT_TOKEN')
        os.environ['YQL_TOKEN'] = sdk2.Vault.data('ROBOT_EDDL_ANALYST_YQL_TOKEN')
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'

        edadeal_funnels_binary = sdk2.Resource.find(
            resource_type=EDADEAL_FUNNELS_TOOL,
            state=ctr.State.READY,
        ).first()
        edadeal_funnels_binary_path = str(sdk2.ResourceData(edadeal_funnels_binary).path)

        run_python_udf_binary = sdk2.Resource.find(
            resource_type=Analytics.RUN_PYTHON_UDF,
            state=ctr.State.READY,
        ).first()
        run_python_udf_binary_path = str(sdk2.ResourceData(run_python_udf_binary).path)

        logging.info('Using EdadealfunnelsTool binary created on: ' + edadeal_funnels_binary.created.strftime('%Y-%m-%d'))
        logging.info('Using run_python_udf binary created on: ' + run_python_udf_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', run_python_udf_binary_path, edadeal_funnels_binary_path, cmd_args, '=======\n\n')

        with sdk2.helpers.ProcessLog(self, logger='Edadealfunnels') as pl:
            sp.check_call(
                [run_python_udf_binary_path, edadeal_funnels_binary_path] + cmd_args,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )

        logging.info("Successfully finished process execution")
