# -*- coding: utf-8 -*-
"""Prepare RetailGeoReach tables."""

import os
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess as sp
from sandbox.projects.Edadeal import Analytics


class RETAIL_GEO_REACH_TOOL(sdk2.Resource):
    """
        RetailerReachByRegionTool binary to prepare YT tables.
    """
    any_arch = True
    executable = True
    releasers = ['antonka', 'adsergeev']


class RetailGeoReach(sdk2.Task):
    """Prepare RetailGeoReach tables."""
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 14400

        start_date = sdk2.parameters.String(
            'Log date to start calculations from.',
            default=''
        )
        end_date = sdk2.parameters.String(
            'Log date to calculate until.',
            default=''
        )
        shop_table = sdk2.parameters.String(
            'Shops table.',
            default=''
        )
        event_table = sdk2.parameters.String(
            'Events table.',
            default=''
        )
        retail_names_table = sdk2.parameters.String(
            'Retail names table.',
            default=''
        )
        result_table = sdk2.parameters.String(
            'Result output table.',
            default=''
        )
        geobase_table = sdk2.parameters.String(
            'Geobase (match edadeal to yandex) table.',
            default=''
        )

    def on_execute(self):
        cmd_args = []

        for key, value in (
            ['--start-date', self.Parameters.start_date],
            ['--end-date', self.Parameters.end_date],
            ['--shop-table', self.Parameters.shop_table],
            ['--event-table', self.Parameters.event_table],
            ['--result-table', self.Parameters.result_table],
            ['--geobase-table', self.Parameters.geobase_table],
            ['--retail-names-table', self.Parameters.retail_names_table],
        ):
            if value:
                cmd_args.extend([key, value])

        os.environ['YT_TOKEN'] = sdk2.Vault.data('ROBOT_EDDL_ANALYST_YT_TOKEN')
        os.environ['YQL_TOKEN'] = sdk2.Vault.data('ROBOT_EDDL_ANALYST_YQL_TOKEN')
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'
        os.environ['robot_wiz_quality'] = sdk2.Vault.data('robot_wiz_quality')

        retail_geo_reach_binary_resources = sdk2.Resource.find(
            resource_type=RETAIL_GEO_REACH_TOOL,
            state='READY'
        ).limit(3)
        tool_last_binary = sorted(list(retail_geo_reach_binary_resources), key=lambda x: x.created)[-1]
        tool_last_binary_path = str(sdk2.ResourceData(tool_last_binary).path)

        run_python_udf_binary_resources = sdk2.Resource.find(
            resource_type=Analytics.RUN_PYTHON_UDF,
            state='READY'
        ).limit(3)
        prog_last_binary = sorted(list(run_python_udf_binary_resources), key=lambda x: x.created)[-1]
        prog_last_binary_path = str(sdk2.ResourceData(prog_last_binary).path)

        logging.info('Using RetailerReachByRegionTool binary created on: ' + tool_last_binary.created.strftime('%Y-%m-%d'))
        logging.info('Using run_python_udf binary created on: ' + prog_last_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', prog_last_binary_path, tool_last_binary_path, cmd_args, '=======\n\n')

        with sdk2.helpers.ProcessLog(self, logger='RetailGeoReach') as pl:
            sp.check_call([prog_last_binary_path, tool_last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT)

        logging.info("Successfully finished process execution")
