# -*- coding: utf-8 -*-
"""Calculate edadeal sessions table and write to YT."""

import os
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers.process import subprocess as sp
from sandbox.projects.Edadeal import Analytics


class EDADEAL_SESSIONS_TOOL(sdk2.Resource):
    """
        EdadealSessionsTool binary to prepare YT sessions-logs.
    """
    any_arch = True
    executable = True
    releasers = ['antonka', 'adsergeev']


class EdadealSessions(sdk2.Task):
    """Prepare EdadealSessions tables."""
    class Requirements(sdk2.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 21600

        start_date = sdk2.parameters.String(
            'Date to start calculations from.',
            default=''
        )
        end_date = sdk2.parameters.String(
            'Date to calculate until.',
            default=''
        )
        log_start_date_gap = sdk2.parameters.String(
            'Extra log tables to take from left.',
            default=''
        )
        log_end_date_gap = sdk2.parameters.String(
            'Extra log tables to take from right.',
            default=''
        )
        create_link = sdk2.parameters.Bool(
            'Create a symlink for the last date.',
            default=True
        )
        results_folder = sdk2.parameters.String(
            'Folder to store sessions in.',
            default=''
        )

    def on_execute(self):
        cmd_args = []

        # String parameters
        for key, value in (
            ['--start-date', self.Parameters.start_date],
            ['--end-date', self.Parameters.end_date],
            ['--log-start-date-gap', self.Parameters.log_start_date_gap],
            ['--log-end-date-gap', self.Parameters.log_end_date_gap],
            ['--results-folder', self.Parameters.results_folder],
        ):
            if value:
                cmd_args.extend([key, value])

        # Boolean parameters (flags)
        for key, value in (
            ['--create-link', self.Parameters.create_link],
        ):
            if value:
                cmd_args.append(key)

        os.environ['YT_TOKEN'] = sdk2.Vault.data('ROBOT_EDDL_ANALYST_YT_TOKEN')
        os.environ['YQL_TOKEN'] = sdk2.Vault.data('ROBOT_EDDL_ANALYST_YQL_TOKEN')
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'

        edadeal_sessions_binary_resources = sdk2.Resource.find(
            resource_type=EDADEAL_SESSIONS_TOOL,
            state='READY'
        ).limit(3)
        tool_last_binary = sorted(list(edadeal_sessions_binary_resources), key=lambda x: x.created)[-1]
        tool_last_binary_path = str(sdk2.ResourceData(tool_last_binary).path)

        run_python_udf_binary_resources = sdk2.Resource.find(
            resource_type=Analytics.RUN_PYTHON_UDF,
            state='READY'
        ).limit(3)
        prog_last_binary = sorted(list(run_python_udf_binary_resources), key=lambda x: x.created)[-1]
        prog_last_binary_path = str(sdk2.ResourceData(prog_last_binary).path)

        logging.info('Using EdadealSessionsTool binary created on: ' + tool_last_binary.created.strftime('%Y-%m-%d'))
        logging.info('Using run_python_udf binary created on: ' + prog_last_binary.created.strftime('%Y-%m-%d'))
        logging.info('=======\n\n', prog_last_binary_path, tool_last_binary_path, cmd_args, '=======\n\n')

        with sdk2.helpers.ProcessLog(self, logger='EdadealSessions') as pl:
            sp.check_call([prog_last_binary_path, tool_last_binary_path] + cmd_args, stdout=pl.stdout, stderr=sp.STDOUT)

        logging.info("Successfully finished process execution")
