# coding: utf-8

from datetime import date
import logging
import os
import time

from sandbox import sdk2
from sandbox import common
from sandbox.sandboxsdk import environments
import sandbox.common.types.task as ctt
from sandbox.common.types import client as ctc
from sandbox.projects.porto import BuildPortoLayer
from sandbox.common.share import skynet_get
from sandbox.projects.Edadeal.resource_types import PortoLayerEdadeal


class BuildCookerPortoLayer(sdk2.Task):
    BUILD_PORTO_LAYER_ATTRIBUTES = {'dependencies', 'name', 'parent_layer', 'stack', 'unpacked_size'}

    SKYGET_TRIES_NUM = 5
    SKYGET_INITIAL_DELAY = 5
    SKYGET_BACKOFF_FACTOR = 2

    EDADEAL_JUNK_PATH = '//home/edadeal/junk'
    SANDBOX_TASK_ID_ATTR_NAME = 'sandbox_task_id'
    COOKER_VERSION = 'cooker_version'

    class Parameters(sdk2.Task.Parameters):
        def resource_types_with_prefixes(*args):
            return [r for r in sdk2.Resource if any(r.name.startswith(arg) for arg in args)]

        with sdk2.parameters.Group('Base parameters') as main_block:
            layer_type = sdk2.parameters.String('Layer type', required=True, default='PORTO_LAYER')
            layer_name = sdk2.parameters.String('Layer name', required=True, default='python3.7_yt_cooker')
            with sdk2.parameters.String('Compress', multiline=True, required=True) as compress:
                for choice in ('tar.gz', 'tar.xz', 'tar'):
                    if choice == 'tar.xz':
                        compress.values[choice] = compress.Value(default=True)
                    else:
                        compress.values[choice] = None
            parent_layer = sdk2.parameters.Resource(
                'Parent layer', required=False, resource_type=resource_types_with_prefixes('PORTO_LAYER'))
            parent_layer_type = sdk2.parameters.String('Parent Layer type', required=True, default='PORTO_LAYER_SEARCH_UBUNTU_BIONIC')
            script_url = sdk2.parameters.String(
                'Setup scripts URLs (arcadia:/path[@rev] or https://...)', multiline=True, required=False)
            script_env = sdk2.parameters.Dict('Environment variables', required=False)
            merge_all_layers = sdk2.parameters.Bool('Merge all layers', required=False, default=True)
            space_limit = sdk2.parameters.Integer('Max disk usage (MB)', required=False, default=18000)
            memory_limit = sdk2.parameters.Integer('Max memory usage (MB)', required=False, default=4096)
            debug_build = sdk2.parameters.Bool('Debug build', required=False)
            start_os = sdk2.parameters.Bool('Start OS for scripts', required=True, default=True)
            post_os_script_url = sdk2.parameters.String(
                'Post OS setup scripts URLs', multiline=True, required=True)

        with sdk2.parameters.Group('YT upload parameters') as yt_block:
            upload_to_yt = sdk2.parameters.Bool('Upload layer to YT', required=False)
            add_postfix_with_id_and_date = sdk2.parameters.Bool(
                'Add to uploading layer name sandbox task id and current date', required=False)
            cluster = sdk2.parameters.String('Target YT cluster', required=False)
            yt_path = sdk2.parameters.String('YT cypress path', required=False)
            yt_link_path = sdk2.parameters.String('YT link cypress path', required=False)
            yt_token_vault_name = sdk2.parameters.String('YT token vault name', required=False,
                                                         default='edadeal-yt-token')
            package_version = sdk2.parameters.String('Package version', required=False)

        with sdk2.parameters.Group('Additional parameters') as add_block:
            fail_on_start = sdk2.parameters.Bool(
                'Fail on start (is used when one of teamcity subtasks was failed '
                'but sandbox build was not stopped)', required=False)
            bootstrap = sdk2.parameters.Url('Bootstrap for base layer', required=False)
            bootstrap_two = sdk2.parameters.Url('Second stage bootstrap', required=False)
            output_resource_id = sdk2.parameters.Integer(
                'ID of resource to be filled with output layer', required=False, default=None)

    class Requirements(sdk2.Task.Requirements):
        environments = (environments.PipEnvironment('yandex-yt'),)
        client_tags = ctc.Tag.GENERIC

    description = 'Build and upload cooker porto layer'

    def _check_for_success(self, task_id):
        if self.server.task[task_id].read()['status'] != 'SUCCESS':
            raise common.errors.TaskFailure('Error in task {}'.format(task_id))

    def _get_yt_client(self):
        from yt.wrapper import YtClient
        return YtClient(self.Parameters.cluster, token=sdk2.Vault.data(self.Parameters.yt_token_vault_name))

    def _upload_to_yt(self):
        yt_client = self._get_yt_client()
        if self.Parameters.add_postfix_with_id_and_date:
            yt_path = self.Parameters.yt_path[:-len(self.Parameters.compress) - 1]
            yt_modified_path = '{}_{}_{}.{}'.format(
                yt_path, self.id, date.today().strftime('%d_%m_%Y'), self.Parameters.compress)
        else:
            yt_modified_path = self.Parameters.yt_path
        with open('{}.{}'.format(self.Parameters.layer_name, self.Parameters.compress), 'r') as f:
            yt_client.create('map_node', os.path.dirname(yt_modified_path), recursive=True, ignore_existing=True)
            yt_client.write_file(yt_modified_path, f)
            yt_client.set_attribute(yt_modified_path, self.COOKER_VERSION, self.Parameters.package_version)
        if self.Parameters.yt_link_path:
            # if link does not exist we create it somehow and then rewrite to avoid race
            yt_client.link(self.EDADEAL_JUNK_PATH, self.Parameters.yt_link_path,
                           ignore_existing=True, recursive=True)
            with yt_client.Transaction():
                unresolved_yt_link_path = '{}&'.format(self.Parameters.yt_link_path)
                yt_client.lock(unresolved_yt_link_path, waitable=True, wait_for=10 * 1000)
                if yt_client.has_attribute(unresolved_yt_link_path, self.SANDBOX_TASK_ID_ATTR_NAME):
                    sandbox_task_id = yt_client.get_attribute(
                        unresolved_yt_link_path, self.SANDBOX_TASK_ID_ATTR_NAME)
                    if sandbox_task_id >= int(self.id):
                        return
                yt_client.link(yt_modified_path, self.Parameters.yt_link_path, recursive=True, force=True)
                yt_client.set_attribute(unresolved_yt_link_path, self.SANDBOX_TASK_ID_ATTR_NAME, int(self.id))

    def _get_resource(self, layer_resource_id):
        from api.copier.errors import CopierError

        delay = self.SKYGET_INITIAL_DELAY
        for i in xrange(self.SKYGET_TRIES_NUM):
            try:
                skynet_get(layer_resource_id, '.')
                return
            except CopierError as e:
                retries_left = self.SKYGET_TRIES_NUM - i - 1
                logging.exception('Cannot download resource %s: %s; retries left: %s',
                                  layer_resource_id, e, retries_left)
                if retries_left != 0:
                    time.sleep(delay)
                    delay *= self.SKYGET_BACKOFF_FACTOR
        raise Exception('Cannot download resource %s, no retries left', layer_resource_id)

    def on_execute(self):
        if self.Parameters.fail_on_start:
            raise Exception('Some of teamcity subtasks failed')
        if not self.Context.build_layer_task_id:
            if self.Parameters.upload_to_yt:
                yt_required_params = (
                    self.Parameters.cluster, self.Parameters.yt_path, self.Parameters.yt_token_vault_name)
                if not all(yt_required_params):
                    raise common.errors.TaskFailure('Not all parameters are set for YT uploading')
                if not self.Parameters.yt_path.endswith(self.Parameters.compress):
                    raise common.errors.TaskFailure('Sandbox layer and yt path extensions differ')

            script_env = dict(self.Parameters.script_env)
            layer_build_params = {
                BuildPortoLayer.ParentLayerType.name: self.Parameters.parent_layer_type,
                BuildPortoLayer.ParentLayer.name: self.Parameters.parent_layer,
                BuildPortoLayer.LayerType.name: self.Parameters.layer_type,
                BuildPortoLayer.LayerName.name: self.Parameters.layer_name,
                BuildPortoLayer.Compress.name: self.Parameters.compress,
                BuildPortoLayer.BootstrapUrl.name: self.Parameters.bootstrap,
                BuildPortoLayer.Bootstrap2Url.name: self.Parameters.bootstrap_two,
                BuildPortoLayer.ScriptUrl.name: self.Parameters.script_url,
                BuildPortoLayer.Script2Url.name: self.Parameters.post_os_script_url,
                BuildPortoLayer.ScriptEnv.name: script_env,
                BuildPortoLayer.StartOS.name: self.Parameters.start_os,
                BuildPortoLayer.MergeLayers.name: self.Parameters.merge_all_layers,
                BuildPortoLayer.DebugBuild.name: self.Parameters.debug_build,
                BuildPortoLayer.SpaceLimit.name: self.Parameters.space_limit,
                BuildPortoLayer.MemoryLimit.name: self.Parameters.memory_limit,
                BuildPortoLayer.OutputResourceID.name: self.Parameters.output_resource_id,
                'kill_timeout': 5 * 60 * 60
            }

            logging.debug('BUILD_PORTO_LAYER build params: %s', str(layer_build_params))
            task_class = sdk2.Task['BUILD_PORTO_LAYER']
            layer_build_task = task_class(
                self,
                description=self.Parameters.description,
                **{
                    key: value.id if isinstance(value, sdk2.Resource) else value
                    for key, value in layer_build_params.iteritems()
                }
            ).enqueue()
            self.Context.build_layer_task_id = layer_build_task.id

            logging.debug('BUILD_PORTO_LAYER build task: %d', layer_build_task.id)

            raise sdk2.WaitTask(
                [self.Context.build_layer_task_id], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
        else:
            self._check_for_success(self.Context.build_layer_task_id)
            layer_resource = sdk2.Resource.find(task_id=self.Context.build_layer_task_id,
                                                type=self.Parameters.layer_type).first()
            self._get_resource(layer_resource.skynet_id)
            new_layer_resource = PortoLayerEdadeal(
                self, layer_resource.description, '{}.{}'.format(self.Parameters.layer_name, self.Parameters.compress))
            sdk2.ResourceData(new_layer_resource).ready()
            for attr_name, attr_value in layer_resource:
                logging.debug('attr name: %s, value: %s', attr_name, attr_value)
                if attr_name in self.BUILD_PORTO_LAYER_ATTRIBUTES:
                    setattr(new_layer_resource, attr_name, attr_value)
            layer_resource.ttl = 1
            if self.Parameters.upload_to_yt:
                self._upload_to_yt()
