import logging

from sandbox import sdk2
from sandbox.common import rest
from sandbox.common import errors
from sandbox.common.types import task as ctt

from sandbox.projects.common.wizard.current_production import get_resource_id_from_task
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.components.configs.entitysearch import EntitySearchBackendCfg
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common.differ import coloring
from sandbox.projects.EntitySearch import resource_types
from sandbox.projects.EntitySearch import common as es_common


CLIENTS = [
    'nmeta',
    'antiobject',
    'yanswer',
    'video',
]

GRPC_CLIENTS = [
    'antiobject',
    'yanswer',
]


MARKS = [
    'old',
    'new',
]


class AcceptanceEntitysearchExecutable(sdk2.Task):

    class Context(sdk2.Context):

        binary_res_id = dict()
        config_res_id = dict()
        fresh_res_id = None
        data_res_id = dict()
        queries_task_id = None
        queries_res_id = dict()
        component_name = None
        augment_url = dict()
        shooters_started = False
        # shooter_tasks = {'new': {'antiobject': 31}, 'old': {'antiobject': 31}}  # {m: dict() for m in MARKS}
        shooter_tasks = {m: dict() for m in MARKS}
        app_host_shooter_task = {m: dict() for m in MARKS}

    class Parameters(sdk2.Parameters):

        kill_timeout = 14000

        binary_old = sdk2.parameters.Task('Entitysearch binary (old)', task_type='ENTITYSEARCH_BINARIES_BUILD', required=False)
        binary_new = sdk2.parameters.Task('Entitysearch binary (new)', task_type='ENTITYSEARCH_BINARIES_BUILD', required=False)
        config_old = sdk2.parameters.Task('Entitysearch config (old)', task_type='ENTITYSEARCH_CONFIG_BUILD', required=False)
        config_new = sdk2.parameters.Task('Entitysearch config (new)', task_type='ENTITYSEARCH_CONFIG_BUILD', required=False)
        data_old = sdk2.parameters.Task('Entitysearch data (old)', task_type='ENTITYSEARCH_DATA_BUILD', required=False)
        data_new = sdk2.parameters.Task('Entitysearch data (new)', task_type='ENTITYSEARCH_DATA_BUILD', required=False)

        fresh = sdk2.parameters.Task('Entitysearch fresh', task_type='ENTITYSEARCH_FRESH_BUILD', required=False)

        queries_to_shoot = sdk2.parameters.Task('Plain text queries', task_type='ENTITYSEARCH_LOGS', required=False)
        dolbilka_requests_limit = sdk2.parameters.Integer('Requests limit', default=300000)
        dolbilka_sessions = sdk2.parameters.Integer('Number of sessions', default=5)
        dolbilka_augment_url_old = sdk2.parameters.String('String to append to each request URL (old)', default='')
        dolbilka_augment_url_new = sdk2.parameters.String('String to append to each request URL (new)', default='')

        grpc_client_requests_limit = sdk2.parameters.Integer('Requests limit for grpc clients', default=300000)
        grpc_sessions = sdk2.parameters.Integer('Sessions per second for grpc clients', default=100)

    def get_queries_resource_for_client(self, task, client):
        return sdk2.Resource['PLAIN_TEXT_QUERIES'].find(task=task, attrs={'client': client}).limit(1).first()

    def init_context(self):
        self.Context.binary_res_id['old'] = self.get_specified_or_prod_resource(resource_types.ENTITY_SEARCH_EXECUTABLE, self.Parameters.binary_old)
        self.Context.binary_res_id['new'] = self.get_specified_or_prod_resource(resource_types.ENTITY_SEARCH_EXECUTABLE, self.Parameters.binary_new)
        self.Context.config_res_id['old'] = self.get_specified_or_prod_resource(resource_types.ENTITY_SEARCH_CONFIG, self.Parameters.config_old)
        self.Context.config_res_id['new'] = self.get_specified_or_prod_resource(resource_types.ENTITY_SEARCH_CONFIG, self.Parameters.config_new)
        self.Context.data_res_id['old'] = self.get_specified_or_prod_resource(resource_types.ENTITY_SEARCH_DATA, self.Parameters.data_old)
        self.Context.data_res_id['new'] = self.get_specified_or_prod_resource(resource_types.ENTITY_SEARCH_DATA, self.Parameters.data_new)
        self.Context.augment_url['old'] = self.Parameters.dolbilka_augment_url_old
        self.Context.augment_url['new'] = self.Parameters.dolbilka_augment_url_new
        self.Context.fresh_res_id = self.get_specified_or_prod_resource(resource_types.ENTITY_SEARCH_FRESH, self.Parameters.fresh)
        self.Context.queries_res_id = dict()
        for client in CLIENTS:
            task = self.Parameters.queries_to_shoot
            if not task:
                task_candidates = sdk2.Task.find(sdk2.Task['ENTITYSEARCH_LOGS'], status=(ctt.Status.SUCCESS)).order(-sdk2.Task.id).limit(30)
                for task_candidate in task_candidates:
                    if self.get_queries_resource_for_client(task_candidate, client):
                        task = task_candidate
                        break
                if not task:
                    raise Exception("Failed to find ENTITYSEARCH_LOGS with requests from client %s" % client)

            self.Context.queries_res_id[client] = self.get_queries_resource_for_client(task, client).id

        self.Context.component_name = EntitySearchBackendCfg.name

    def on_execute(self):
        with self.memoize_stage.start_shooters():
            self.init_context()
            if not self.Context.shooters_started:
                self.start_shooters()

        self.check_shooters()

    def check_shooters(self):
        failed_tasks_ids = []

        for client in CLIENTS:
            for stand in MARKS:
                child_id = self.Context.shooter_tasks[stand][client]
                if not es_common.check_task_status_succeed(task_id=child_id, raise_not_ok=False):
                    failed_tasks_ids.append(child_id)

        for client in GRPC_CLIENTS:
            for stand in MARKS:
                child_id = self.Context.app_host_shooter_task[stand][client]
                if not es_common.check_task_status_succeed(task_id=child_id, raise_not_ok=False):
                    failed_tasks_ids.append(child_id)

        if failed_tasks_ids:
            joined_ids = ', '.join(
                lb.task_link(task_id=task_id)
                for task_id in failed_tasks_ids
            )
            raise errors.TaskFailure("Few sub_tasks haven't finished successfully, their ids: " + joined_ids)

    def start_shooters(self):
        subtasks = []
        for mark in MARKS:
            for client in CLIENTS:
                kwargs = {
                    'entitysearch_binary': self.Context.binary_res_id[mark],
                    'entitysearch_config': self.Context.config_res_id[mark],
                    'entitysearch_data': self.Context.data_res_id[mark],
                    'fresh': self.Context.fresh_res_id,
                    'entitysearch_start_timeout': 6000,
                    'turn_off_fetcher': True,
                    'dolbilka_executor_sessions': self.Parameters.dolbilka_sessions,
                    'dolbilka_executor_requests_limit': self.Parameters.dolbilka_requests_limit,
                    'dolbilka_executor_max_simultaneous_requests': 12,
                    'dolbilka_augment_url': self.Context.augment_url[mark],
                    'reqs': self.Context.queries_res_id[client],
                }
                performance_task_class = sdk2.Task['ENTITYSEARCH_SHARP_SHOOTER']
                logging.info('Starting shooter task for stand={}, client={}'.format(mark, client))
                subtask = performance_task_class(
                    self,
                    description="Shooter task for entitysearch executable priemka (stand={}, client={})".format(mark, client),
                    owner=self.Parameters.owner,
                    priority=self.Parameters.priority,
                    notifications=self.Parameters.notifications,
                    **kwargs
                ).enqueue()
                logging.info('Subtask started')
                subtasks.append(subtask)
                self.Context.shooter_tasks[mark][client] = subtask.id

                if client in GRPC_CLIENTS:
                    app_host_shooter_task_class = sdk2.Task['ENTITY_SEARCH_APPHOST_SHOOTER']
                    logging.info('Starting apphost shooter task for stand={}, client={}'.format(mark, client))
                    kwargs = {
                        'entitysearch_binary': self.Context.binary_res_id[mark],
                        'entitysearch_config': self.Context.config_res_id[mark],
                        'entitysearch_data': self.Context.data_res_id[mark],
                        'fresh': self.Context.fresh_res_id,
                        'requests_limit': self.Parameters.grpc_client_requests_limit,
                        'sessions_per_second': self.Parameters.grpc_sessions,
                        'turn_off_fetcher': True,
                        'entitysearch_start_timeout': 6000,
                        'sources': client.upper()
                    }
                    subtask = app_host_shooter_task_class(
                        self,
                        description="Apphost grpc shooter task for entitysearch executable priemka (stand={}, client={})".format(mark, client),
                        owner=self.Parameters.owner,
                        priority=self.Parameters.priority,
                        notifications=self.Parameters.notifications,
                        **kwargs
                    ).enqueue()
                    logging.info('Subtask started')
                    subtasks.append(subtask)
                    self.Context.app_host_shooter_task[mark][client] = subtask.id

        self.Context.shooters_started = True
        raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def get_specified_or_prod_resource(self, resource_type, task=None):
        if task:
            return get_resource_id_from_task(task.id, resource_type)
        else:
            return sdk2.Resource.find(resource_type, attrs={'released': 'stable'}).limit(1).first().id

    @property
    def footer(self):
        rest_client = rest.Client()
        foot = []
        foot.extend(self._get_performance_foot(rest_client))
        return foot

    def _get_performance_foot(self, rest_client):
        head = [{"key": "empty", "title": "&nbsp;"}]
        body = {"empty": [
            "Old (max rps)", "New (max rps)",
            "<strong>Diff (between max)</strong>",
            "Grpc client Old (fail rate)", "Grpc client Old (max response)",
            "Grpc client New (fail_rate)", "Grpc client New (max response)"
        ]}
        for client in CLIENTS:
            head.append({'key': client, 'title': client})
            body[client] = []
            child_rps = []

            for stand in MARKS:
                child_id = self.Context.shooter_tasks[stand][client]
                if not child_id:
                    body[client].append("<em style='color:Cyan'>no child</em>")
                else:
                    child_info = rest_client.task.read(
                        id=self.Context.shooter_tasks[stand][client],
                        fields='status,context.max_rps',
                        children=True,
                        hidden=True,
                        limit=1,
                    ).get("items", [{}])[0]
                    max_rps = child_info['context.max_rps']
                    child_rps.append(max_rps)
                    body[client].append(
                        lb.task_link(
                            task_id=child_id,
                            link_name=child_id,
                        ) + " {}".format(max_rps)
                    )
            try:
                diff = 100 * (child_rps[1] - child_rps[0]) / child_rps[0]
            except Exception:
                diff = '-'
            body[client].append(coloring.color_diff(diff, max_diff=-3))

            if client in GRPC_CLIENTS:
                for stand in MARKS:
                    child_id = self.Context.app_host_shooter_task[stand][client]
                    if not child_id:
                        body[client].append("<em style='color:Cyan'>no child</em>")
                    else:
                        child_info = rest_client.task.read(
                            id=child_id,
                            fields='status,context.response_quantiles,context.info',
                            children=True,
                            hidden=True,
                            limit=1,
                        ).get("items", [{}])[0]
                        logging.info('Radomir custom footer. Find child id in {}, {}'.format(stand, client))
                        fail_rate = child_info['context.info']['Fail rate']
                        max_response = child_info['context.response_quantiles']['Max response size']
                        body[client].append(
                            lb.task_link(
                                task_id=child_id,
                                link_name=child_id,
                            ) + " {}".format(fail_rate)
                        )
                        body[client].append(max_response)


        return [{
            'helperName': '',
            'content': {'<h3>Performance tests</h3>': {
                "header": head,
                "body": body
            }}
        }]
