import jinja2
import logging
import os
import struct

from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process
from sandbox.sdk2.paths import make_folder
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.math import percentile
from sandbox.projects.app_host import resources as app_host_resources
from sandbox.projects.resource_types import TASK_CUSTOM_LOGS

from sandbox.projects.common.wizard.providers import EntitySearchProvider, StartTimeoutParameter, UseMemoryMapping, TurnOffFetcher, DisableKvsaasMainDb
from sandbox.projects.common.wizard.providers import EntitySearchBinary
from sandbox.projects.common.wizard.providers import EntitySearchConfig
from sandbox.projects.common.wizard.providers import EntitySearchData
from sandbox.projects.common.wizard.providers import Fresh
from sandbox.projects.common.wizard.providers import NerData

from sandbox.projects.EntitySearch import resource_types

import sandbox.common.types.task as ctt


PRODUCTION_SERVICE = 'sas-production-entitysearch-yp'


def get_prod_resource(resource_type):
    return sdk2.Resource.find(resource_type, attrs={'released': 'stable'}).limit(1).first()


def get_plan_resource(task, source):
    return sdk2.Resource['ENTITY_SEARCH_APP_HOST_SHOOTER_PLAN'].find(task=task, attrs={'source': source}).limit(1).first()


class EntitySearchApphostShooter(sdk2.Task):
    """ A task to shoot in app_host for YANSWER and ANTIOBJECT(grpc client)"""

    class Requirements(sdk2.Requirements):
        ram = 300 * 1024  # 500 gb ram
        disk_space = 400 * 1024  # 400 gb disk

    class Parameters(sdk2.Parameters):
        kill_timeout = 21600  # 6 hours
        entitysearch_binary = sdk2.parameters.Resource('es binary', resource_type=resource_types.ENTITY_SEARCH_EXECUTABLE, required=False)
        entitysearch_config = sdk2.parameters.Resource('es config', resource_type=resource_types.ENTITY_SEARCH_CONFIG, required=False)
        entitysearch_data = sdk2.parameters.Resource('Ontodb data', resource_type=resource_types.ENTITY_SEARCH_DATA, required=False)
        ner = sdk2.parameters.Resource('Ner', resource_type=resource_types.ENTITY_SEARCH_NER_DATA, required=False, default=1872308461)
        entitysearch_start_timeout = sdk2.parameters.Integer('Start timeout (sec)', required=False, default=1500)
        turn_off_fetcher = sdk2.parameters.Bool('Turn off fetcher', required=False, default=False)
        fresh = sdk2.parameters.Resource('Fresh data', resource_type=resource_types.ENTITY_SEARCH_FRESH, required=False)

        sessions_per_second = sdk2.parameters.Integer('Number of sessions', default=100)
        queries_to_shoot = sdk2.parameters.Resource('Grpc client plan, empty to autodetect', resource_type=resource_types.ENTITY_SEARCH_APP_HOST_SHOOTER_PLAN, required=False)
        requests_limit = sdk2.parameters.Integer('Requests limit', default=300000)
        grpc_client = sdk2.parameters.Resource('Grpc client executable, empty to autodetect', resource_type=app_host_resources.AppHostGrpcClientExecutable, required=False)
        with sdk2.parameters.RadioGroup('Sources', required=True) as sources:
            sources.values.YANSWER = sources.Value('YANSWER', default=True)
            sources.values.ANTIOBJECT = sources.Value('ANTIOBJECT')

    def on_execute(self):
        source = self.Parameters.sources
        if self.Parameters.queries_to_shoot is None:
            task_candidates = sdk2.Task.find(sdk2.Task['ENTITY_SEARCH_APPHOST_REQUEST_SAMPLER'], status=(ctt.Status.SUCCESS)).order(-sdk2.Task.id).limit(30)
            queries_to_shoot = None
            for task in task_candidates:
                logging.info('Find in task = {}'.format(task.id))
                resource = get_plan_resource(task, source)
                if resource:
                    queries_to_shoot = resource
                    break
            if queries_to_shoot is None:
                raise TaskFailure("Error. Not autodetect queries to shoot from ENTITY_SEARCH_APPHOST_REQUEST_SAMPLER")
        else:
            queries_to_shoot = self.Parameters.queries_to_shoot

        context = dict()
        for (param_resource, resource_name, resource_type) in [(self.Parameters.entitysearch_binary, EntitySearchBinary.name, resource_types.ENTITY_SEARCH_EXECUTABLE),
                                                               (self.Parameters.entitysearch_config, EntitySearchConfig.name, resource_types.ENTITY_SEARCH_CONFIG),
                                                               (self.Parameters.entitysearch_data, EntitySearchData.name, resource_types.ENTITY_SEARCH_DATA),
                                                               (self.Parameters.ner, NerData.name, resource_types.ENTITY_SEARCH_NER_DATA),
                                                               (self.Parameters.fresh, Fresh.name, resource_types.ENTITY_SEARCH_FRESH)]:
            if param_resource is None:
                context[resource_name] = get_prod_resource(resource_type).id
            else:
                context[resource_name] = param_resource.id
        nanny_token = sdk2.Vault.data('robot-ontodb', 'nanny-oauth-token')

        context[StartTimeoutParameter.name] = StartTimeoutParameter.default_value
        context[UseMemoryMapping.name] = UseMemoryMapping.default_value
        context[TurnOffFetcher.name] = TurnOffFetcher.default_value
        context[DisableKvsaasMainDb.name] = DisableKvsaasMainDb.default_value

        context['parent_nanny_service'] = 'sas-production-entitysearch-yp'
        context['entitysearch_start_timeout'] = self.Parameters.entitysearch_start_timeout
        context['turn_off_fetcher'] = self.Parameters.turn_off_fetcher

        port = str(8897)
        context['custom_cmd'] = '{binary_path} --data {data_path} --fresh {fresh_path} --ner {ner_path} --cfg {cfg_path}' \
                                ' -p {port} --apphost-port {apphost_port} --apphost-threads {apphost_threads} --apphost-grpc-port ' + port
        with EntitySearchProvider.from_task_context(context, nanny_token=nanny_token) as provider:
            logging.info('Run entity search')

            prefix = source.lower()

            if self.Parameters.grpc_client is None:
                grpc_client = str(sdk2.ResourceData(get_prod_resource(app_host_resources.AppHostGrpcClientExecutable)).path)
            else:
                grpc_client = str(sdk2.ResourceData(self.Parameters.grpc_client).path)

            plan = str(sdk2.ResourceData(queries_to_shoot).path)
            request_count = str(self.Parameters.requests_limit)
            response_dir = make_folder('tmp_response')
            sessions_count = str(self.Parameters.sessions_per_second)

            name = 'grpc_client_eror_log'
            desc = 'Stderror of grpc_client'
            error_resource = TASK_CUSTOM_LOGS(self, desc, name)
            with open(str(sdk2.ResourceData(error_resource).path), 'w') as error:
                logging.info('Start shoot via grpc_client')
                cmd = [grpc_client, '--format', 'bin', '--connection-timeout', '100', '--request-count', request_count,
                                 '--sessions-per-second', sessions_count, '--response-out-dir', response_dir, '--plan', plan,
                                 '--address', 'localhost:{}/{}'.format(port, prefix)]
                logging.info('Start with ' + ' '.join(cmd))
                try:
                    run_process(cmd, stderr=error)
                except SandboxSubprocessError as subprocess_error:
                    raise TaskFailure("Grpc client died. Error message: " + subprocess_error.message + ". Inspect grpc_client_eror_log")

            if not provider.alive():
                raise TaskFailure('ES is dead. Inspect entitysearch.err.txt in logs')

        response_sizes = []
        with open(os.path.join(response_dir, 'data'), 'rb') as data:
            byte_size = data.read(4)
            while byte_size:
                size = struct.unpack('I', byte_size)[0]
                response_sizes.append(size)
                data.read(size)
                byte_size = data.read(4)
                response_sizes.sort()

        self.Context.response_quantiles = {'50 Percentile response size': percentile(response_sizes, 0.5),
                                           '95 Percentile response size': percentile(response_sizes, 0.95),
                                           'Max response size': response_sizes[-1]}

        quantile_prefixes = ['Q(0.5)', 'Q(0.95)', 'Q(0.99)']
        self.Context.time_quantiles = dict()
        with open(str(sdk2.ResourceData(error_resource).path), 'r') as error:
            for line in reversed(error.readlines()):
                line = line.strip()
                if line.startswith('sessions'):
                    requests = int(line.split('\t')[0].split('=')[-1])
                    sucesses = int(line.split('\t')[1].split('=')[-1])
                    fails = int(line.split('\t')[2].split('=')[-1])
                    fail_rate = fails * 1.0 / requests
                    break
                else:
                    for quantile_prefix in quantile_prefixes:
                        if line.startswith(quantile_prefix):
                            self.Context.time_quantiles[quantile_prefix + ' ms'] = '{:0.2f}'.format(float(line.split('=')[-1][:-1]) * 1000)
                            break

        self.Context.info = {'Total requests': requests, 'Total sucesses': sucesses,
                             'Total fails': fails, 'Fail rate': fail_rate}

    @sdk2.footer()
    def display_footer(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), extensions=['jinja2.ext.do'])
        return env.get_template('footer.html').render(
            {
                'info': self.Context.info,
                'response_quantiles': self.Context.response_quantiles,
                'time_quantiles': self.Context.time_quantiles
            }
        )
