import requests
import logging
import urllib2
import json
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import apihelpers
import sandbox.common.types.task as ctt
from sandbox.projects.EntitySearch import resource_types
from sandbox.projects.common.apihelpers import get_last_released_resource

BUILD_TYPE = 'ENTITY_SEARCH_EXECUTABLE'
CONFIG_TYPE = 'ENTITY_SEARCH_CONFIG'


def get_queries_resource(task, client='nmeta'):
    return sdk2.Resource['PLAIN_TEXT_QUERIES'].find(task=task, attrs={'client': client}).limit(1).first()


def find_es_resource(task, resource_type):
    for resource in apihelpers.list_task_resources(task.id):
        if resource.type == resource_type:
            return resource.id


class EntitySearchTrunkTest(sdk2.Task):
    """ A task to test entitysearch from trunk on production request"""

    class Requirements(sdk2.Requirements):
        dns = DnsType.DNS64

    class Parameters(sdk2.Parameters):
        kill_timeout = 21600  # 6 hours
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Svn url for arcadia", required=True)
        debug_tool = sdk2.parameters.String('debug tool parameter for es', default='sanitizer')
        dolbilka_requests_limit = sdk2.parameters.Integer('Requests limit for entitysearch', default=300000)
        apphost_requests_limit = sdk2.parameters.Integer('Requests limit for serp_object', default=300000)
        slack_web_hook_vault = sdk2.parameters.String('Vault with Webhook url for Slack notification',
                                                      default_value='ya-ontology-slack-token', required=True)

    def make_slack_url(self):
        webhook = sdk2.Vault.data(self.Parameters.slack_web_hook_vault)
        url = 'https://hooks.slack.com/services/{webhook}'.format(webhook=webhook)
        return url

    def _notify_slack(self, message):
        url = self.make_slack_url()
        requests.post(
            url,
            headers={'Content-Type': 'application/json'},
            data=json.dumps({
                'user_name': 'incoming-webhook',
                'text': message,
                'channel': '#es_trunk_test'
            }))

    def build_binary(self, subtasks):
        build_task_class = sdk2.Task['ENTITYSEARCH_BINARIES_BUILD']
        build_task = build_task_class(
            self,
            description='Build entitysearch bin',
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            build_type='debug',
            use_arc_instead_of_aapi=True,
            aapi_fallback=True,
            sanitize='address'
        ).enqueue()
        self.Context.build_binary_task_id = build_task.id
        subtasks.append(build_task.id)
        logging.info('Build entitysearch binary')

    def build_config(self, subtasks):
        build_task_class = sdk2.Task['ENTITYSEARCH_CONFIG_BUILD']
        build_task = build_task_class(
            self,
            description='Build entitysearch config',
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
        ).enqueue()
        self.Context.build_config_task_id = build_task.id
        subtasks.append(build_task.id)
        logging.info('Build entitysearch config')

    def find_queries_es_logs(self):
        task_candidates = sdk2.Task.find(sdk2.Task['ENTITYSEARCH_LOGS'], status=(ctt.Status.SUCCESS)).order(-sdk2.Task.id).limit(30)
        for task in task_candidates:
            if get_queries_resource(task):
                return task

    def shoot_es(self, es_binary, es_config, queries_task):
        kwargs = {
            'entitysearch_binary': es_binary,
            'entitysearch_config': es_config,
            'entitysearch_start_timeout': 10000,
            'debug_tool': self.Parameters.debug_tool,
            'dolbilka_executor_requests_limit': self.Parameters.dolbilka_requests_limit,
            'reqs': get_queries_resource(queries_task).id
        }
        performance_task_class = sdk2.Task['ENTITYSEARCH_SHARP_SHOOTER']
        shooter_task = performance_task_class(
            self,
            description='Shooter task for check entitysearch binary',
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        self.Context.shoot_es_task_id = shooter_task.id
        logging.info('Sharp shooter started')

        del kwargs['reqs']
        del kwargs['dolbilka_executor_requests_limit']
        kwargs['requests_limit'] = self.Parameters.apphost_requests_limit
        del kwargs['debug_tool']

        apphost_grpc_task_class = sdk2.Task['ENTITY_SEARCH_APPHOST_SHOOTER']

        kwargs['sources'] = 'YANSWER'
        yanswer_shooter_task = apphost_grpc_task_class(
            self,
            description='Shooter task for check entitysearch binary in yanswer via grpc_client',
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        self.Context.shoot_yanswer_task_id = yanswer_shooter_task.id
        logging.info('Yanswer grpc shooter started')

        kwargs['sources'] = 'ANTIOBJECT'
        antiobject_shooter_task = apphost_grpc_task_class(
            self,
            description='Shooter task for check entitysearch binary in antiobject via grpc_client',
            owner=self.Parameters.owner,
            priority=self.Parameters.priority,
            notifications=self.Parameters.notifications,
            **kwargs
        ).enqueue()
        self.Context.shoot_antiobject_task_id = antiobject_shooter_task.id
        logging.info('AntiObject grpc shooter started')

        raise sdk2.WaitTask([self.Context.shoot_es_task_id, self.Context.shoot_yanswer_task_id, self.Context.shoot_antiobject_task_id],
                             ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def on_break(self, prev_status, status):
        task_url = 'https://sandbox.yandex-team.ru/task/{}/view'.format(self.id)
        message = '<!channel>, EntitySearchTrunkTest failed with broken state. Something was wrong, for additional information see {}'.format(task_url)
        self._notify_slack(message)

    def on_failure(self, prev_status):
        task_url = 'https://sandbox.yandex-team.ru/task/{}/view'.format(self.id)
        message = '<!channel>, EntitySearchTrunkTest failed with failure state. Something was wrong, for additional information see {}'.format(task_url)
        self._notify_slack(message)

    def on_execute(self):
        task_url = 'https://sandbox.yandex-team.ru/task/{}/view'.format(self.id)
        with self.memoize_stage.on_start():
            self._notify_slack('EntitySearchTrunkTest started. Run on arcadia url = {}, See status on {}'.format(self.Parameters.checkout_arcadia_from_url, task_url))

            subtasks = []
            self.build_binary(subtasks)
            self.build_config(subtasks)

            queries_task = self.find_queries_es_logs()
            if queries_task is None:
                raise TaskFailure('Not find task ENTITYSEARCH_LOGS with requests for client nmeta')
            self.Context.queries_task_id = queries_task.id
            logging.info('find queries task')

            raise sdk2.WaitTask(subtasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

        with self.memoize_stage.shoot_es():
            build_task = sdk2.Task[self.Context.build_binary_task_id]
            if build_task.status in ctt.Status.Group.BREAK or build_task.status == ctt.Status.FAILURE:
                raise TaskFailure("Can't build entitysearch binary from {} in task https://sandbox.yandex-team.ru/task/{}/view".format(self.Parameters.checkout_arcadia_from_url, build_task.id))

            es_binary = find_es_resource(build_task, BUILD_TYPE)
            if es_binary is None:
                raise TaskFailure('Not find {} in task https://sandbox.yandex-team.ru/task/{}/view'.format(BUILD_TYPE, build_task.id))

            build_config_task = sdk2.Task[self.Context.build_config_task_id]
            if build_config_task in ctt.Status.Group.BREAK or build_config_task == ctt.Status.FAILURE:
                raise TaskFailure("Can't get entitysearch config from {}, in task https://sandbox.yandex-team.ru/task/{}/view".format(self.Parameters.checkout_arcadia_from_url, build_config_task.id))

            es_config = find_es_resource(build_config_task, CONFIG_TYPE)
            if es_config is None:
                raise TaskFailure('Not find {} in task https://sandbox.yandex-team.ru/task/{}/view'.format(CONFIG_TYPE, build_config_task.id))

            queries_task = sdk2.Task[self.Context.queries_task_id]

            self.shoot_es(es_binary, es_config, queries_task)

        for (task_id, client) in [(self.Context.shoot_es_task_id, 'ENTITY_SEARCH'),
                                  (self.Context.shoot_yanswer_task_id, 'YANSWER'), (self.Context.shoot_antiobject_task_id, 'ANTIOBJECT')]:
            shooter_task = sdk2.Task[task_id]
            if shooter_task.status in ctt.Status.Group.BREAK or shooter_task.status == ctt.Status.FAILURE:
                raise TaskFailure('Shooter with binary from trunk failed on production requests with client {}. Task https://sandbox.yandex-team.ru/task/{}/view'.format(client, task_id))

        logging.info('Test finished, everything is ok')
        self._notify_slack('EntitySearchTrunkTest finished. Everything is ok. See additional information in {}'.format(task_url))
