# -*- coding: utf-8 -*-

import os
import logging

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.ssh import Key
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn


CARDMERGE_MARKUP_NODE = '//home/dict/ontodb/cardmerge_learn/markup'


class EntitySearchUpdateCardmergeMarkup(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 128

        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    class Parameters(sdk2.Parameters):
        # common parameters
        kill_timeout = 600
        disk_space = 128

        # notifications
        notifications = [
            sdk2.Notification(
                [ctt.Status.SUCCESS, ctt.Status.EXCEPTION, ctt.Status.FAILURE, ctt.Status.Group.BREAK],
                ['entity-search-notifications', 'danshalak'],
                ctn.Transport.EMAIL
            )
        ]

        # custom parameters
        markup_arcadia_path = sdk2.parameters.String(
            'Path to resource files with markup to update',
            default='junk/danshalak/marks',
            required=True,
        )

        markup_file_name = sdk2.parameters.String(
            'File with markup to update',
            required=True,
        )

        markup_data_table = sdk2.parameters.String(
            'Markup data table',
            required=True
        )

        dry_run = sdk2.parameters.Bool('Dry run', default=False)

    def patch_markup_file(self, markup_file_path, markup):
        with open(markup_file_path, 'rw+') as markup_file:
            lines = markup_file.readlines()
            logging.info('{} before patch \n{}'.format(markup_file_path, lines))

            new_lines = []
            for line in markup:
                left = line['inputValues']['left']
                right = line['inputValues']['right']
                mark = 1 if line['outputValues']['result'] == 'yes' else 0
                weight = line['probability']
                new_lines.append('{}\t{}\t{}\t{}'.format(left, right, mark, weight))

            markup_file.write('\n'.join(new_lines))

            logging.info('{} after patch \n{}'.format(markup_file_path, lines + new_lines))

    def commit_updated_markup_file(self, markup):
        logging.info('markup={}'.format(markup))
        markup_files_path = sdk2.Path('markup_resources').absolute()
        logging.info('Checking out svn path self.Parameters.markup_file_name')
        Arcadia.checkout(
            Arcadia.trunk_url(os.path.join(self.Parameters.markup_arcadia_path)),
            markup_files_path
        )

        logging.info('Checked out path is {}'.format(markup_files_path))
        markup_file_path = os.path.join(str(markup_files_path), self.Parameters.markup_file_name)
        self.patch_markup_file(markup_file_path, markup)
        with Key(self, 'robot-ontodb', 'robot-ontodb-ssh-key'):
            Arcadia.commit(
                markup_file_path,
                'Autoupdating markup\' file {}'.format(self.Parameters.markup_file_name),
                user='robot-ontodb'
            )
            logging.info('patched markup file was commited')

    def on_execute(self):
        from yt.wrapper import YtClient
        yt_token = sdk2.Vault.data('robot-ontodb', 'robot-ontodb-yt-token')
        yt = YtClient('hahn', yt_token)

        if not self.Parameters.dry_run:
            self.commit_updated_markup_file(yt.read_table(self.Parameters.markup_data_table))
