# -*- coding: utf-8 -*-

import os
import logging
import json
from collections import defaultdict

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.ssh import Key
from sandbox.common.types import task as ctt
from sandbox.common.types import notification as ctn


CARDMERGE_LEARN = '//home/dict/ontodb/cardmerge_learn'
CARDMERGE_LEARN_DAILY = '//home/dict/ontodb/cardmerge_learn/daily'
CARDMERGE_LEARN_PRODUCTION = '//home/dict/ontodb/cardmerge_learn/production'

# берем словарь и заполняем его списками значений метрик для каждого пула
# считаем среднее + дисперсию
# смотрим насколько мы отклонились сейчас
# переносим в директорию продакшн, делаем коммит ресурса

DEDUP_METRICS_TABLE = 'deduplication_results'
PRECISE_METRICS_TABLE = 'precise_results'
F_MEASURE = 'F-Measure'
PRECISION = 'Precision'
RECALL = 'Recall'


def read_metrics(yt, table, metrics):
    for row in yt.read_table(table, raw=False):
        pool = metrics[row['pool']]
        pool[F_MEASURE] = row[F_MEASURE]


def calculate_cardmerge_acceptance_metrics(yt, daily_cardmerge_dir, baseline_cardmerge_dir, pools_of_interest):
    baseline_metrics = defaultdict(lambda: defaultdict(list))
    new_metrics = defaultdict(lambda: defaultdict(list))

    read_metrics(yt, os.path.join(baseline_cardmerge_dir, PRECISE_METRICS_TABLE), baseline_metrics)
    read_metrics(yt, os.path.join(daily_cardmerge_dir, PRECISE_METRICS_TABLE), new_metrics)

    result = defaultdict(lambda: defaultdict(dict))

    pools_of_interest = pools_of_interest.split(',')
    for (pool, metrics) in baseline_metrics.iteritems():
        if pool in pools_of_interest:
            metric_result = {
                'fm_baseline_value': metrics[F_MEASURE],
                'fm_new_value': new_metrics[pool][F_MEASURE],
                'result': False
            }

            if metric_result['fm_new_value'] >= metric_result['fm_baseline_value']:
                metric_result['result'] = True

            result[pool] = metric_result

    return result


def check_cardmerge_acceptance(context, yt, daily_cardmerge_dir, baseline_cardmerge_dir, pools_of_interest):
    result = calculate_cardmerge_acceptance_metrics(yt, daily_cardmerge_dir, baseline_cardmerge_dir, pools_of_interest)
    logging.info(
        'Acceptance results: \nDaily directory {daily}\nBaseline directory {baseline}\nResult\n{results}'.format(
            daily=daily_cardmerge_dir,
            baseline=baseline_cardmerge_dir,
            results=json.dumps(result, indent=2)
        )
    )

    fails = {}

    for pool, metrics in result.iteritems():
        if not metrics['result']:
            fails[pool] = metrics

    if fails:
        # set fail to task context
        context.acceptance_result = {'result': False}

        # send to paste and notify to slack
        raise TaskFailure(
            (
                'Acceptance failed.\n'
                'Artifacts \nhttps://yt.yandex-team.ru/hahn/navigation?path={}\n'
                'Failed deduplication pools: \n{}'
            ).format(daily_cardmerge_dir, json.dumps(fails, indent=2))
        )
    else:
        context.acceptance_result = {'result': True}


class OntodbCardmergeModel(sdk2.Resource):
    auto_backup = True
    ttl = 'inf'


class EntitySearchUpdateCardmergeResources(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 128

        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    class Parameters(sdk2.Parameters):
        # common parameters
        kill_timeout = 600
        disk_space = 128

        # notifications
        notifications = [
            sdk2.Notification(
                [ctt.Status.SUCCESS, ctt.Status.EXCEPTION, ctt.Status.FAILURE, ctt.Status.Group.BREAK],
                ['entity-search-notifications', 'danshalak'],
                ctn.Transport.EMAIL
            )
        ]

        # custom parameters
        cardmerge_resources_arcadia_path = sdk2.parameters.String(
            'Path to cardmerge/resources directory',
            default='junk/danshalak/cardmerge/resources',
            required=True
        )

        daily_cardmerge_dir = sdk2.parameters.String(
            'Cardmerge daily build node (YT)',
            required=True
        )

        baseline_cardmerge_dir = sdk2.parameters.String(
            'Cardmerge daily baseline build node (YT)',
            required=True
        )

        pools_of_interest = sdk2.parameters.String(
            'List of metric pools to compare. CSV list of pools',
            required=True
        )

        model_file_name = sdk2.parameters.String(
            'Model file name',
            default='matrixnet.info',
            required=True
        )

        dry_run = sdk2.parameters.Bool('Dry run', default=False)

    def patch_ya_make(self, ya_make_path, resource_id):
        new_ya_make_content = []
        with open(ya_make_path) as ya_make:
            lines = ya_make.readlines()
            logging.info('ya.make before patch \n{}'.format(lines))

            for line in lines:
                if ' OUT {})'.format(self.Parameters.model_file_name) in line:
                    new_ya_make_content.append('FROM_SANDBOX(FILE {} OUT {})\n'.format(resource_id, self.Parameters.model_file_name))
                else:
                    new_ya_make_content.append(line)

            logging.info('ya.make after patch \n{}'.format(new_ya_make_content))

        with open(ya_make_path, 'w') as mutable_ya_make:
            mutable_ya_make.write(''.join(new_ya_make_content))

    def create_new_cardmerge_resource(self, yt, model_on_yt, model_file):
        logging.info('creating resource with model file {}'.format(model_on_yt))
        resource = OntodbCardmergeModel(
            self,
            'Cardmerge model resource {}'.format(model_file),
            model_file
        )

        resource_data = sdk2.ResourceData(resource)

        resource_data.path.write_bytes(
            yt.read_file(
                os.path.join(self.Parameters.daily_cardmerge_dir, self.Parameters.model_file_name)
            ).read()
        )

        resource_data.ready()
        logging.info('resource is ready')
        return resource.id

    def commit_new_cardmerge_resource(self, resource_id):
        logging.info('resource_id = {}'.format(resource_id))
        cardmerge_resources_path = sdk2.Path('resources').absolute()
        logging.info('Checking out svn path self.Parameters.cardmerge_resources_arcadia_path')
        Arcadia.checkout(Arcadia.trunk_url(self.Parameters.cardmerge_resources_arcadia_path), cardmerge_resources_path)
        logging.info('checked out path is {}'.format(cardmerge_resources_path))
        ya_make_path = os.path.join(str(cardmerge_resources_path), 'ya.make')
        self.patch_ya_make(ya_make_path, resource_id)
        with Key(self, 'robot-ontodb', 'robot-ontodb-ssh-key'):
            Arcadia.commit(str(cardmerge_resources_path), 'autoupdating cardmerge resources ya.make SKIP_CHECK SKIP_REVIEW', user='robot-ontodb')
            logging.info('patched ya.make was commited')

    def move_daily_cardmerge_dir_to_stable(self, yt):
        logging.info('moving {} to stable builds'.format(self.Parameters.daily_cardmerge_dir))
        yt.move(
            self.Parameters.daily_cardmerge_dir,
            os.path.join(CARDMERGE_LEARN_PRODUCTION, self.Parameters.daily_cardmerge_dir.split('/')[-1]),
            recursive=True
        )

    def on_execute(self):
        from yt.wrapper import YtClient
        yt_token = sdk2.Vault.data('robot-ontodb', 'robot-ontodb-yt-token')
        yt = YtClient('hahn', yt_token)

        check_cardmerge_acceptance(self.Context, yt, self.Parameters.daily_cardmerge_dir, self.Parameters.baseline_cardmerge_dir, self.Parameters.pools_of_interest)

        resource_id = self.create_new_cardmerge_resource(
            yt=yt,
            model_on_yt=os.path.join(self.Parameters.daily_cardmerge_dir, self.Parameters.model_file_name),
            model_file=self.Parameters.model_file_name
        )

        if not self.Parameters.dry_run:
            self.commit_new_cardmerge_resource(resource_id)
            self.move_daily_cardmerge_dir_to_stable(yt)
