# -*- coding: utf-8 -*-

import logging
import os

from sandbox.projects.EntitySearch import resource_types
from sandbox import sdk2
import sandbox.sdk2.helpers

from sandbox.sdk2.helpers import subprocess
from sandbox.projects.common.wizard.current_production import get_current_production_task_id
from sandbox.projects.common.wizard.current_production import get_resource_id_from_task, get_current_production_resource_id

import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
from sandbox.common.types.client import Tag


class EntitysearchFreshConvertorComparison(sdk2.Task):
    ''' Creates two different fresh versions by reconverting given fresh with two specified convertors '''

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = Tag.Group.LINUX & ~Tag.LXC
        disk_space = 1024 * 16
        ram = 1024 * 16

    class Context(sdk2.Context):
        production_service = 'sas-production-entitysearch-yp'
        prod_bin_task_id = None
        nanny_oauth_token = None

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600  # 1 hour

        binaries_task_type = 'ENTITYSEARCH_BINARIES_BUILD'
        bin_descr = 'Built binaries for convertor {0} (uses production if not specified)'

        binaries_build_1 = sdk2.parameters.Task(
            bin_descr.format(1), task_type=binaries_task_type, required=False
        )

        binaries_build_2 = sdk2.parameters.Task(
            bin_descr.format(2), task_type=binaries_task_type, required=False
        )

        fresh_task_type = 'ENTITYSEARCH_FRESH_BUILD'
        fresh_task = sdk2.parameters.Task(
            'Built fresh dir (uses production if not specified)', task_type=fresh_task_type, required=False
        )

    def get_nanny_oauth_token(self):
        if self.Context.nanny_oauth_token is None:
            self.Context.nanny_oauth_token = sdk2.Vault.data('robot-ontodb', 'nanny-oauth-token')

        return self.Context.nanny_oauth_token

    def get_prod_bin_task_id(self):
        if self.Context.prod_bin_task_id is None:
            logging.info('requesting es production binaries task id')
            self.Context.prod_bin_task_id = get_current_production_task_id(
                self.Context.production_service,
                resource_types.ENTITY_SEARCH_EXECUTABLE,
                self.get_nanny_oauth_token()
            )

        logging.info('production es binaries task id is ' + str(self.Context.prod_bin_task_id))
        return self.Context.prod_bin_task_id

    def get_specified_or_prod_convertor(self, binaries_build=None):
        return get_resource_id_from_task(
            binaries_build.id if binaries_build else self.get_prod_bin_task_id(),
            resource_types.ENTITY_SEARCH_CONVERTER
        )

    def get_specified_or_prod_fresh(self):
        if self.Parameters.fresh_task:
            return get_resource_id_from_task(
                self.Parameters.fresh_task.id,
                resource_types.ENTITY_SEARCH_FRESH,
            )
        else:
            return get_current_production_resource_id(
                self.Context.production_service,
                resource_types.ENTITY_SEARCH_FRESH,
                self.get_nanny_oauth_token()
            )

    def get_raw_main_delta_resource(self, fresh_resource):
        self.Context.ONTODB_MAIN_DELTA_VERSION_KEY = 'ontodb_main_delta_ver'
        self.Context.main_delta_version = getattr(fresh_resource, self.Context.ONTODB_MAIN_DELTA_VERSION_KEY, None)

        # TODO: change for new delta using
        return sdk2.Resource.find(
            type=resource_types.ONTODB_CARDS_MAIN_DELTA,
            state=ctr.State.READY,
            attrs={'resource_processed_to_fresh': True, self.Context.ONTODB_MAIN_DELTA_VERSION_KEY: self.Context.main_delta_version}
        ).first()

    def get_convertor_path_by_res_id(self, convertor_res_id):
        return str(sdk2.ResourceData(sdk2.Resource.find(
            type=resource_types.ENTITY_SEARCH_CONVERTER,
            state=ctr.State.READY,
            id=int(convertor_res_id)
        ).first()).path)

    def create_fresh_dir_copy(self, index):
        fresh_dir = '_fresh_{}_'.format(index)
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("copy fresh")) as pl:
            cp_command = 'mkdir -p {1} && cp -r {0} {1}'.format(self.Context.fresh_resource_path, fresh_dir)
            subprocess.check_call(cp_command, shell=True, stderr=pl.stdout, stdout=pl.stdout)

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("make writeable")) as pl:
            subprocess.check_call('chmod -R +w {} '.format(fresh_dir), shell=True, stderr=pl.stdout, stdout=pl.stdout)

        return os.path.abspath(fresh_dir + '/' + os.path.basename(self.Context.fresh_resource_path))

    def create_paths(self, fresh_resource, raw_main_delta_resource):
        self.Context.fresh_resource_path = str(sdk2.ResourceData(fresh_resource).path)
        self.Context.raw_main_delta_resource_path = str(sdk2.ResourceData(raw_main_delta_resource).path)

        self.Context.convertor_1_path = self.get_convertor_path_by_res_id(self.Context.convertor_1_res_id)
        self.Context.convertor_2_path = self.get_convertor_path_by_res_id(self.Context.convertor_2_res_id)

        self.Context.fresh_dir_1_path = self.create_fresh_dir_copy(1)
        self.Context.fresh_dir_2_path = self.create_fresh_dir_copy(2)

    def convert_raw_main_delta(self, convertor_path, fresh_dir_path):
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("convert fresh")) as pl:
            subprocess.check_call(
                '{0} -s {1} -D {2} -n main_delta --codec none'.format(convertor_path, self.Context.raw_main_delta_resource_path, fresh_dir_path),
                shell=True, stderr=pl.stdout, stdout=pl.stdout
            )

    def produce_fresh_resource(self, descr, new_fresh_dir_path, convertor_res_id):
        new_fresh_resource = resource_types.ENTITY_SEARCH_FRESH(
            self,
            descr,
            new_fresh_dir_path,
            releasable=False,
            auto_backup=False,
            ttl='inf'
        )
        setattr(new_fresh_resource, self.Context.ONTODB_MAIN_DELTA_VERSION_KEY, self.Context.main_delta_version)
        setattr(new_fresh_resource, 'description', descr)
        setattr(new_fresh_resource, 'convertor_resource_id', int(convertor_res_id))
        setattr(new_fresh_resource, 'source_fresh_resource_id', int(self.Context.fresh_res_id))
        setattr(new_fresh_resource, 'raw_main_delta_resource_id', int(self.Context.raw_main_delta_res_id))
        sdk2.ResourceData(new_fresh_resource).ready()

    def on_execute(self):
        self.Context.convertor_1_res_id = self.get_specified_or_prod_convertor(self.Parameters.binaries_build_1)
        self.Context.convertor_2_res_id = self.get_specified_or_prod_convertor(self.Parameters.binaries_build_2)
        self.Context.fresh_res_id = self.get_specified_or_prod_fresh()

        fresh_resource = sdk2.Resource.find(
            type=resource_types.ENTITY_SEARCH_FRESH,
            state=ctr.State.READY,
            id=int(self.Context.fresh_res_id)
        ).first()

        raw_main_delta_resource = self.get_raw_main_delta_resource(fresh_resource)
        self.Context.raw_main_delta_res_id = raw_main_delta_resource.id

        self.create_paths(fresh_resource, raw_main_delta_resource)

        self.convert_raw_main_delta(self.Context.convertor_1_path, self.Context.fresh_dir_1_path)
        self.convert_raw_main_delta(self.Context.convertor_2_path, self.Context.fresh_dir_2_path)

        self.produce_fresh_resource('fresh converted with convertor 1', self.Context.fresh_dir_1_path, self.Context.convertor_1_res_id)
        self.produce_fresh_resource('fresh converted with convertor 2', self.Context.fresh_dir_2_path, self.Context.convertor_2_res_id)

        logging.info('done')

        self.Context.nanny_oauth_token = None
