# -*- coding: utf-8 -*-
import os
import shlex
import shutil
import subprocess as sp

from sandbox import sdk2
from sandbox.sdk2 import yav

import sandbox.common.types.client as ctc
from sandbox.projects.common.nanny import nanny

from sandbox.projects.EntitySearch import resource_types
from sandbox.projects.EntitySearch.common.current_production import get_resource_task_id_from_service
from sandbox.projects.common.wizard.current_production import get_current_production_task_id


PRODUCTION_ENTITYSEARCH_NANNY_SERVICE = 'sas-production-entitysearch-yp'


class EntitySearchMainDbBuild(nanny.ReleaseToNannyTask2, sdk2.Task):
    """Выполняет конвертацию таблички в траи и поддерживает логику релизов совместно принятых ресурсов"""

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Convert db parameters') as convert_db_parameters:
            server_name = sdk2.parameters.String('Yt server', default='hahn.yt.yandex.net')
            table = sdk2.parameters.String(
                'Table name or version',
                default='home/dict/ontodb/ver/main/production/all_cards_final'
            )

            converter_res = sdk2.parameters.Resource(
                'Converter',
                resource_type=resource_types.ENTITY_SEARCH_CONVERTER,
            )

            result_prefix = sdk2.parameters.String('Result prefix', default='main')

            with sdk2.parameters.String('Codec', multiline=True) as codec:
                codec.values.default = codec.Value('default', default=True)
                codec.values.none = 'none'

            split_card = sdk2.parameters.Bool("'split-card' flag from converter (only for main base)", default=False)
            use_production_yt_quota_for_tmp = sdk2.parameters.Bool(
                "Override '--tmp-table-prefix' to production quota YT directory",
                default=True,
            )

            with sdk2.parameters.String('Yt Log Level', multiline=True) as yt_log_level:
                yt_log_level.values.DEBUG = 'DEBUG'
                yt_log_level.values.INFO = yt_log_level.Value('INFO', default=True)
                yt_log_level.values.WARNING = 'WARNING'
                yt_log_level.values.ERROR = 'ERROR'

        db_version = sdk2.parameters.String('Db version (for resource annotation)', required=False)

        with sdk2.parameters.Group('Extra resources acceptance (in Helix)') as extra_resources_acceptance:
            prepare_extra_resources = sdk2.parameters.Bool('Prepare extra resources (for daily run, in Helix)', default=False)

            with prepare_extra_resources.value[True]:
                main_delta_task_for_acceptance = sdk2.parameters.Task(
                    'Main delta Preparation Task to make accepted (default is current production version)',
                    task_type='ENTITY_SEARCH_MAIN_DELTA_RESOURCE_PREPARATION',
                )

                es_shard_task_for_acceptance = sdk2.parameters.Task(
                    'ES Data Build Task to make accepted (default is latest pre-built version)',
                    task_type='ENTITYSEARCH_DATA_BUILD',
                )

                include_delta_into_archive = sdk2.parameters.Bool(
                    'Include selected delta into archive',
                    description='As accepted_main_delta',
                    default=True,
                )

                release_with_extra_resources = sdk2.parameters.Bool("Release other tasks' resources with releasing this task", default=False)

        with sdk2.parameters.Output:
            with sdk2.parameters.Group('Resources to release') as resources_to_release:
                main_db = sdk2.parameters.Resource('Main db archive', resource_type=resource_types.ENTITY_SEARCH_CONVERT_DB_ARCHIVE, required=True)
                are_resources_prepared = sdk2.parameters.Bool("Are extra resources prepared", default=False)
                are_resources_selected = sdk2.parameters.Bool("Are extra resources selected", default=False)
                selected_main_delta = sdk2.parameters.Resource('Selected main delta', resource_type=resource_types.ENTITY_SEARCH_MAIN_DELTA)
                entitysearch_data = sdk2.parameters.Resource('Shard', resource_type=resource_types.ENTITY_SEARCH_DATA)

    class Requirements(sdk2.Task.Requirements):
        cores = 4
        disk_space = 128 * 1024  # 128 GB
        ram = 144 * 1024  # 144 GB
        client_tags = ctc.Tag.Group.LINUX & ctc.Tag.SSD

    @property
    def yt_token(self):
        return yav.Secret("sec-01daxfc7h9w8hbmcxys8950djg").data()["secret"]

    @property
    def nanny_token(self):
        return yav.Secret("sec-01dgz9bx7jwgxb3e0dpxt3r8f5").data()["robot-ontodb-nanny-token"]

    @property
    def converter_result_dir(self):
        return '{}_db'.format(self.Parameters.result_prefix)

    def init_yt_env(self):
        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_LOG_LEVEL'] = self.Parameters.yt_log_level
        os.environ['YT_TOKEN'] = self.yt_token
        os.environ['YT_CLIENT_TIMEOUT'] = '300'

    def get_converter_path(self):
        converter = None
        if self.Parameters.converter_res:
            converter = self.Parameters.converter_res
        else:
            task_id = get_resource_task_id_from_service(
                service=PRODUCTION_ENTITYSEARCH_NANNY_SERVICE,
                resource_type=resource_types.ENTITY_SEARCH_EXECUTABLE,
                nanny_oauth_token=self.nanny_token
            )

            converter = sdk2.Resource.find(
                task_id=task_id,
                type=resource_types.ENTITY_SEARCH_CONVERTER
            ).first()

        self.Context.converter_id = converter.id
        es_converter_resource = sdk2.ResourceData(converter)
        return str(es_converter_resource.path.absolute())

    def convert(self):
        table = self.Parameters.table
        table_name = 'home/dict/ontodb/ver/main/0.{}/all_cards_final'.format(table) if table.isdigit() else table

        result_dir = self.converter_result_dir
        external_dir = os.path.join(result_dir, 'external')
        os.makedirs(external_dir)

        cmd = '''{cnv_bin} -s {src_table} --server {yt_srv} --card-filter=shown-only {tmp_table_prefix}
                --external {external_dir} -D {result_dir} -n {pref} {codec} {split_card}
        '''.format(
            cnv_bin=self.get_converter_path(),
            src_table=table_name,
            yt_srv=self.Parameters.server_name,
            tmp_table_prefix=(
                '--tmp-table-prefix home/dict/ontodb/entitysearch_convertor'
                if self.Parameters.use_production_yt_quota_for_tmp
                else ''
            ),
            external_dir=external_dir,
            result_dir=result_dir,
            pref=self.Parameters.result_prefix,
            codec='' if self.Parameters.codec == 'default' else ' --codec ' + self.Parameters.codec,
            split_card=' --split-card' if self.Parameters.split_card is True else '',
        )

        with sdk2.helpers.ProcessLog(self, logger='convertor') as pl:
            sp.check_call(shlex.split(cmd), stdout=pl.stdout, stderr=pl.stderr)

    def register_convert_resources(self):
        archive_attrs = {}
        if self.Parameters.db_version:
            archive_attrs['db_version'] = self.Parameters.db_version

        archive_resource = resource_types.ENTITY_SEARCH_CONVERT_DB_ARCHIVE(
            task=self,
            description='archive for {}'.format(self.Parameters.table),
            path=self.converter_result_dir,
            **archive_attrs
        )

        sdk2.ResourceData(archive_resource).ready()
        self.Parameters.main_db = archive_resource

    def on_execute(self):
        with self.memoize_stage.do_prepare_extra_resources(commit_on_entrance=False):
            if self.Parameters.prepare_extra_resources:
                self.select_main_delta_to_accept()
                self.select_entitysearch_data()
                self.Parameters.are_resources_prepared = True
                self.Parameters.are_resources_selected = True

        with self.memoize_stage.do_convert(commit_on_entrance=False):
            self.init_yt_env()
            self.convert()
            self.move_result_resource_files()
            self.register_convert_resources()

    def select_entitysearch_data(self):
        shard_task = None
        if self.Parameters.es_shard_task_for_acceptance:
            shard_task = self.Parameters.es_shard_task_for_acceptance
        else:
            # TODO: change for ReleaseMachine routine
            shard_task = sdk2.Task.find(
                type='ENTITYSEARCH_DATA_BUILD',
                status='RELEASED'
            ).first()

        self.Context.data_build_task_id = shard_task.id
        self.Parameters.entitysearch_data = sdk2.Resource.find(
            task_id=self.Context.data_build_task_id,
            resource_type=resource_types.ENTITY_SEARCH_DATA,
        ).first()

    def select_main_delta_to_accept(self):
        if self.Parameters.main_delta_task_for_acceptance:
            self.Context.main_delta_task_id = self.Parameters.main_delta_task_for_acceptance.id
        else:
            production_task_id = get_current_production_task_id(
                service=PRODUCTION_ENTITYSEARCH_NANNY_SERVICE,
                resource_type=resource_types.ENTITY_SEARCH_MAIN_DELTA,
                nanny_oauth_token=self.nanny_token
            )
            self.Context.main_delta_task_id = sdk2.Task.find(id=production_task_id).first().id

        main_delta_resource = sdk2.Resource.find(
            task_id=self.Context.main_delta_task_id,
            resource_type=resource_types.ENTITY_SEARCH_MAIN_DELTA,
        ).first()

        self.Parameters.selected_main_delta = main_delta_resource

    def move_result_resource_files(self):
        gzt_convert_file = os.path.join(self.converter_result_dir, self.Parameters.result_prefix + '.gzt')
        if os.path.exists(gzt_convert_file):
            os.remove(gzt_convert_file)

        if self.Parameters.selected_main_delta and self.Parameters.include_delta_into_archive:
            main_delta_resource = self.Parameters.selected_main_delta
            main_delta_resource_path = str(sdk2.ResourceData(main_delta_resource).path.absolute())

            for f in ('main_delta.trie', 'main_delta.gzt.bin'):
                shutil.copyfile(
                    src=os.path.join(main_delta_resource_path, f),
                    dst=os.path.join(self.converter_result_dir, 'accepted_' + f),
                )

    def on_release(self, additional_parameters):
        if self.Parameters.prepare_extra_resources and self.Parameters.release_with_extra_resources:
            # TODO: add full release
            pass

            # # TODO: check if not released
            # subj = 'Releasing data build task with main db build '
            # if self.Parameters.db_version:
            #     subj += self.Parameters.db_version + ' '
            #
            # task_link = 'sandbox.yandex-team.ru/task/' + str(self.id)
            # es_data_release_params = {
            #     'task_id': int(self.Context.data_build_task_id),
            #     'type': additional_parameters['release_status'],
            #     'subject': subj + '({})'.format(task_link),
            #     'to': additional_parameters.get('to', []),
            # }
            #
            # # TODO: add logging
            #
            # self.server.release(es_data_release_params)

        super(EntitySearchMainDbBuild, self).on_release(additional_parameters)

    @property
    def release_template(self):
        subj = 'Releasing ' + self.Parameters.description

        if self.Parameters.db_version:
            subj += ' (ver {})'.format(self.Parameters.db_version)

        # TODO: add info about releasing extra resources (?)
        return sdk2.ReleaseTemplate(
            cc=['entity-search-releases'],
            subject=subj,
            message=subj,
        )
