import requests
import logging

from sandbox import sdk2
from sandbox.sdk2 import yav


class HitmanProcessTask():
    HITMAN_URL = 'https://hitman.yandex-team.ru'
    JOB_STATUS_OK = {'SUCCEEDED'}
    JOB_STATUS_FAILED = {'FAILED', 'CANCELED'}
    JOB_WAIT_PERIOD = 3 * 60
    class Parameters(sdk2.Task.Parameters):
        process_id = sdk2.parameters.String('process id', required=True, default='Chimera_runner')

    @property
    def hitman_token(self):
        return yav.Secret('sec-01dgz9a9ygjywhww81at9ys0c4').data()['robot-ontodb-oauth']

    @property
    def headers(self):
        return {
            'Content-Type': 'application/json',
            'Authorization': 'OAuth {}'.format(self.hitman_token),
        }

    @property
    def start_proc_url(self):
        return '/'.join([self.HITMAN_URL, 'api/v1/execution/start', self.Parameters.process_id])

    def start_hitman_process(self, requester, properties, comment):
        logging.info('Start Hitman process, API url %s', self.start_proc_url)
        request_data = {'requester': requester, 'properties': properties, 'comment': comment}
        logging.info('Send data to Hitman: %s', str(request_data))
        resp = requests.post(self.start_proc_url, headers=self.headers, json=request_data)
        logging.info('Hitman response: %s', str(resp.content))
        if 'id' not in resp.json():
            raise Exception("Hitman api response: %s", str(resp.content))
        return resp.json()['id']

    def get_hitman_job(self, job_id):
        url = '/'.join([self.HITMAN_URL, 'api/v2/jobs', job_id])
        logging.info('Get Hitman job info, API url %s', url)
        resp = requests.get(url, headers=self.headers)
        logging.info('Hitman response: %s', str(resp.content))
        return resp.json()

    def get_job_status(self, job_id):
        return self.get_hitman_job(job_id)['status']

    def wait_hitman_job_result(self, job_id):
        status = self.get_job_status(job_id)
        logging.info("Job #{} status {}".format(job_id, status))
        if status in self.JOB_STATUS_OK:
            return True
        elif status in self.JOB_STATUS_FAILED:
            return False
        else:
            logging.info("Waiting for hitman job")
            raise sdk2.WaitTime(self.JOB_WAIT_PERIOD)
